


print("Loading modules...")
import argparse
import os
import datetime
import jpholiday
import glob
import re
from decimal import Decimal
import shutil
import time

# import geopy
from geopy.distance import geodesic

from PIL import Image
from PIL.ExifTags import TAGS

import googlemaps
import polyline

import json

import t2g3lib

print("Loaded.")

BREAK_CHAR = '!'

# property:
GENERATOR_NAME = 't2g3'
GENERATOR_VERSION = 'ver.0.1'
GENERATOR_URL = 'http://pdic.la.coocan.jp/trk2googlemap/'
LOCALTIME_DIFF = 9
JPG_TIMEDIFF = 9
GOOGLE_MAPS_KEY = ""
GOOGLE_MAPS_LANG = "ja"
VELOCITY_UNIT = 5	# 平均速度の時間幅[min]
MAP_WIDTH = 640
MAP_HEIGHT = 480
THUMB_WIDTH = 256
THUMB_HEIGHT = 256
BODY_WEIGHT = 0
LOAD_WEIGHT = 0

MIN_REST_TIME = 0 * 60	# 累積距離・時間の計算対象外とする最小休憩時間[sec] ←これは何のために必要？以前は15分に設定 2023.4.11

REGION_INFLATE_RATIO = 0

Version = 2		# script version

MarkdownMode = False
ShowPhotoDistAlt = True

Width = 800
PhotoColumns = 3
MaxCommentChars = 80	# 画像の下にコメントを表示する最大文字数

# 徒歩用の休憩パラメータ
MaxRestArea = 5	# [m]
MinRestTime = 2 * 60 # [sec]
MaxReturnTime = 2 * 60 # [sec]

# 車用の休憩パラメータ
MaxRestAreaCar = 15	# [m]
MinRestTimeCar = 5 * 60 # [sec]
MaxReturnTimeCar = 2* 60 # [sec]

def setCarMode():
	global MaxRestArea, MinRestTime, MaxReturnTime
	MaxRestArea = MaxRestAreaCar
	MinRestTime = MinRestTimeCar
	MaxReturnTime = MaxReturnTimeCar

# 徒歩 or 自動車の判定parameter
CarVelocity = 15	# [km/h] 平均速度がこれ以上は自動車
CarVelocityTime = 60	# [min] CarVelocityがこの時間以上ある場合は自動車

MinElevPeak = 20	# [m] peakを数えるための最低高度
MinElevUpDown = 3	# [m] 累積上り・下り高低差を計算するための最低高度差

MinElev = 0

MountArriveDistance = 200	# [m] point到着判定の距離

MountPointsFile = 'mounts.txt'
PointsFile = 'points.txt'

AbstractTitle = '概要'

Properties = {}
MPoints = []

DummyPhotoBaseId = 1000	# dummy photo用のbase id

SCRIPT_PATH, name = os.path.split(__file__)

RPhotoNames = []	# comment.txtから取得したphoto nameなど
RTitle = {}
RComment = {}
MarkedFiles = []
MarkedFileComments = {}

PhotoIds = []	# "photoID\d+"
PhotoBaseNames = {}	# "DSC_?\d+"
BottomComment = {}

PhotoIdToIndex = {}	# photo_id -> index of photos

# configuration - switch
PrintPhotoName = False

PropertiesFile = 'properties.txt'
MapsFilesDir = 'maps.files'
PhotoNamesFile = 'template.txt'
CommentFile = 'comment.txt'
PhotoCommentFile = 'photocomment.txt'			# comment.txt作成前に、このファイルからコメントを追加
PhotoMarksFile = 'photomarks.txt'				# markしたファイルの一覧
PhotoMarkCommentFile = 'photomarkcomments.txt'	# photomarks.txtに対応するコメントファイル
CommentTemplFileCar = 'comment-car-tpl.txt'		# sectionを追加した場合は PredefinedSection にも追加すること
CommentTemplFileWalk = 'comment-walk-tpl.txt'	# sectionを追加した場合は PredefinedSection にも追加すること

PhotosMDTemplFile = 'photos-tpl.md'
PhotosMD = 'photos.md'

# working values
CurYear = 0
CurMon = 0
CurDay = 0
CurDate = None
CurDateStr = ''

Verbose = False

PredefinedSection = ['version', 'date', 'prev_date', 'next_date', 'cost', 'kcal', 'yamap', 'coursetime', 'abstract', 'last', 'add', 'comment', 'photo', 'title']

CostHeaderHidden = ''

# Hints
# points : lat/lng/ele/timeの配列
# locations : lat,lngの配列

def main():
	parser = argparse.ArgumentParser(description="trk2gmap")
	parser.add_argument("files", nargs="*")
	parser.add_argument('-md', '--markdown', action='store_true', help='Generate comment/photos files for markdown')
	parser.add_argument('-l', '--load_parsed_data', action='store_true', help='load parsed data from json file(t2g.json)')
	parser.add_argument('-p', '--print_photo_name', action='store_true')
	parser.add_argument('-n', '--not_make_thumbnail', action='store_true')
	parser.add_argument('-c', '--only_comment', action='store_true')
	parser.add_argument('-a', '--no_addr', action='store_true')
	parser.add_argument('-u', '--update_points', action='store_true', help='download and update points.txt file')
	parser.add_argument('-v', '--verbose', action='store_true')
	parser.add_argument('-t', '--test', action='store_true')
	parser.add_argument('-w', '--wait', action='store_true', help='Wait for comment.txt update before going process')
	parser.add_argument('-ok', action='store_true', help='write ok file when finished normally')
	parser.add_argument('-cc', '--calc_course_time', action='store_true', help='Calculate course time from course constant')
	args = parser.parse_args()

	if args.test:
		test(args.files)
		# return

	if args.markdown:
		global MarkdownMode
		MarkdownMode = True

	if args.calc_course_time:
		if len(args.files) < 4:
			print("ERROR: Need four parameters - course_const/distance[km]/total_up[m]/total_down[m].")
			return
		stat = {}
		cc = float(args.files[0])
		stat['dist'] = float(args.files[1])
		stat['total_up'] = int(args.files[2])
		stat['total_down'] = int(args.files[3])
		ct = calcCourseTimeFromCourseConst(stat, cc)
		ct1 = calcCourseTimeFromCourseConst(stat, cc-0.5)
		ct2 = calcCourseTimeFromCourseConst(stat, cc+0.5)
		print(f'{ct.seconds//3600}:{ct.seconds//60%60:02} range: {ct1.seconds//3600}:{ct1.seconds//60%60:02} - {ct2.seconds//3600}:{ct2.seconds//60%60:02}')
		return

	if args.wait:
		st_cmt = os.stat(CommentFile).st_mtime
		st_md = None
		if os.path.exists(PhotosMD):
			st_md = os.stat(PhotosMD).st_mtime
		print("Waiting for "+CommentFile+", "+PhotosMD+" updated...")
		while True:
			mt = os.stat(CommentFile).st_mtime
			if mt != st_cmt:
				break
			if st_md:
				mt = os.stat(PhotosMD).st_mtime
				if mt != st_md:
					break
			time.sleep(1)

	readProperties()

	if args.update_points:
		updatePoints()

	if not GOOGLE_MAPS_KEY:
		print(f"GOOGLE_MAPS_KEY should be defined in {PropertiesFile}.")
		exit(1)

	global Verbose
	Verbose = args.verbose

	if args.print_photo_name:
		global PrintPhotoName
		PrintPhotoName = args.print_photo_name

	make_thumb = not args.not_make_thumbnail

	global Version, CurYear, CurMon, CurDay, CurDate, CurDateStr

	# Read comment file
	global RPhotoNames, RTitle, RComment, MarkedFiles
	r, RPhotoNames, RTitle, RComment, MarkedFiles = t2g3lib.parseCommentFile(CommentFile)
	if r:
		if 'date' in RTitle:
			m = re.match(r'(\d+)\/(\d+)\/(\d+)', RTitle['date'])
			if m:
				CurYear = int(m.group(1))
				CurMon = int(m.group(2))
				CurDay = int(m.group(3))
		if 'version' in RTitle:
			Version = int(RTitle['version'])
			if Version == 0:
				# as default
				pass
			elif Version == 2 or Version == 3:
				# version0仕様に変換
				# RTitle[]が!で始まっていない場合は、'-'行と同じ扱い
				for name in list(RTitle.keys()):
					if name in PredefinedSection:
						continue
					if RTitle[name].startswith('!'):
						RTitle[name] = RTitle[name][1:]
					elif RTitle[name] == '-' or RTitle[name] == '*':
						pass
					else:
						# タイトル無し
						if name not in RComment:
							RComment[name] = RTitle[name]
						else:
							RComment[name] = RTitle[name] + '\n' + RComment[name]
						RTitle[name] = '-'
			else:
				print("Unknown version: ", Version)
				exit(1)
		else:
			Version = 0

	if not CurYear or not CurMon or not CurDay:
		cdir = os.getcwd()
		print("Current Dir:", cdir)
		m = re.match(r'.*(\d\d\d\d)(\d\d)(\d\d)\w*$', cdir)
		if m:
			CurYear = int(m.group(1))
			CurMon = int(m.group(2))
			CurDay = int(m.group(3))
		else:
			print("Warning: current directory is not YYYYMMDD format!!")
			print("         date is not correct if comment.txt file is created.")
			print("Enter correct date(YYYYMMDD).")
			print("If you leave empty, it assumes today's date.")
			now = datetime.datetime.now()
			CurYear = now.year
			CurMon = now.month
			CurDay = now.day
			inp = input()
			if inp != '':
				m = re.match(r'(\d\d\d\d)(\d\d)(\d\d)', inp)
				if m:
					CurYear = int(m.group(1))
					CurMon = int(m.group(2))
					CurDay = int(m.group(3))
				else:
					print("Unknown date format.")
					exit(1)

	CurDate = datetime.datetime(year=CurYear, month=CurMon, day=CurDay)
	CurDateStr = "{}/{}/{}".format(CurYear, CurMon, CurDay)

	if not os.path.exists(MapsFilesDir):
		os.mkdir(MapsFilesDir)

	points = []
	photos = []
	markers = []

	if args.load_parsed_data:
		print("Loading from json file...")
		points, photos, markers = loadInputData()
	else:
		points_in_jpg = False
		for file in args.files:
			base, ext = os.path.splitext(file)
			print(file)
			files = glob.glob(file)
			if ext == '.gpx':
				for file in files:
					print(f'GPX Reading...{file}')
					# parse_gpx(file)
					points += read_gpx(file)
					# locs = pointsToLocs(points)
			elif ext == '.log':
				for file in files:
					print(f'NMEA Reading...{file}')
					points += read_nmea(file, CurDate)
					# locs = pointsToLocs(points)
			elif ext == '.jpg':
				for file in files:
					# parse_jpg(file)
					if args.verbose:
						print("  ", file)
					dt, ori, lat, lng, alt, cmt = get_jpg_info(file)
					# print(dt)
					p = {}
					if dt:
						p['time'] = dt
					dir, name = os.path.split(file)
					base, ext = os.path.splitext(name)
					if lat:
						p['lat'] = lat
						p['lng'] = lng
						points_in_jpg = True
					if alt:
						p['alt'] = alt
					p['name'] = base	# DSC12345
					p['thw'] = THUMB_WIDTH
					p['thh'] = THUMB_HEIGHT
					p['file'] = base + ".jpg"
					p['thumb'] = MapsFilesDir+'/'+base+'-thumb.jpg'
					p['ori'] = ori
					p['cmt'] = cmt
					photos.append(p)

		files = glob.glob('*-marks.kml')
		for file in files:
			print(f"Reading...{file}")
			markers += read_kml(file)
		# print(markers)

	stat = {}
	global MinElev
	global EnvGraphHidden, EnvGraphAllHidden, EnvGraphPressHidden, EnvGraphTempHidden
	EnvGraphHidden = 'hidden'
	EnvGraphAllHidden = 'hidden'
	EnvGraphPressHidden = 'hidden'
	EnvGraphTempHidden = 'hidden'

	rest_debug = False
	if rest_debug:	# test
		dist_cum, alt_min, alt_max = calcDistanceVelocity(points)
		rests, sum_rest_time = getRests(points)
		print(rests)
		global MinElev
		MinElev = MinElevUpDown
		peaks, total_up, total_dn, peak_count, total_up_dst, total_dn_dst, total_up_time, total_dn_time = getPeaks(points, rests, MinElev)		# 累積高低差を求める計算
		print("-- dist_cum={} time={} rest={}".format(dist_cum, (points[-1]['time'] - points[0]['time']).seconds, sum_rest_time))
		exit(0)

	if not args.only_comment and points:
		# 距離、速度を求める
		print("Calculate distance/velocity")
		dist_cum, alt_min, alt_max = calcDistanceVelocity(points)

		# ５分平均の速度を求める
		sum = 0		# sum of velocity values
		sum_ct = 0
		start_time = None
		start_dt = None
		avg_dur = VELOCITY_UNIT	# duration for average [min]
		avg_points = []	# 平均速度情報
		carveltime = 0
		for pt in points:
			cur_dt = roundTime(pt['time'], avg_dur*60)	# avg_dur単位に変換
			if start_dt != None and start_dt != cur_dt:
				apt = {}
				apt['time'] = start_time
				vel = sum / sum_ct
				apt['vel'] = vel
				if vel >= CarVelocity:
					carveltime += VELOCITY_UNIT
				avg_points.append(apt)
				sum = 0
				sum_ct = 0
			if sum_ct == 0:
				start_time = cur_dt
				start_dt = cur_dt
			sum += pt['vel']
			sum_ct += 1
		
		stat['car'] = False
		if carveltime >= CarVelocityTime:
			stat['car'] = True
			setCarMode()
			print("Set car mode.")

		# 実移動距離（休憩時間）を求める
		sum_rest_time = 0
		rests = None
		if not stat['car']:
			rests, sum_rest_time = getRests(points)
			stat['rests'] = rests
			stat['rest_time'] = sum_rest_time

		# 総高低差を求める
		MinElev = MinElevUpDown
		if stat['car']: MinElev = MinElevPeak	# 車だと小さなup/downの計算は不要なので
		peaks, total_up, total_dn, peak_count, total_up_dst, total_dn_dst, total_up_time, total_dn_time = getPeaks(points, rests, MinElev)		# 累積高低差を求める計算
		# peakは粗めで計算
		if not stat['car']:
			peaks, _total_up, _total_dn, peak_count, _total_up_dst, _total_dn_dst, _total_up_time, _total_dn_time = getPeaks(points, rests, MinElevPeak)	# peak/restを求める計算
		stat['peaks'] = peaks
		stat['total_up'] = total_up
		stat['total_down'] = total_dn
		stat['peak_count'] = peak_count
		stat['total_up_dst'] = total_up_dst
		stat['total_down_dst'] = total_dn_dst
		stat['total_up_time'] = total_up_time
		stat['total_down_time'] = total_dn_time
		stat['total_up_avg'] = total_up_dst / total_up_time * 60 * 60 if total_up_time else 0	# km/h
		stat['total_down_avg'] = total_dn_dst / total_dn_time * 60 * 60	if total_dn_time else 0	# km/h

		stat['dist'] = dist_cum
		stat['avgspeed'] = dist_cum / (points[-1]['time'] - points[0]['time']).seconds * 3600
		stat['actualavgspeed'] = dist_cum / ((points[-1]['time'] - points[0]['time']).seconds - sum_rest_time) * 3600
		stat['minalt'] = alt_min
		stat['maxalt'] = alt_max

		# 通過した山を調べる
		if not stat['car']:
			stat['mounts'] = getMounts(points)

		saveStat(stat)

		# data.json
		enc_points = encodePoints(points)
		levels = [8, 1, 1, 2, 2, 2, 3, 1, 2, 1, 2, 3, 1, 2, 2, 1, 3, 2, 4, 1, 2, 2, 2, 3, 2, 2, 2, 0, 3, 1, 2, 2, 3, 4, 1, 2, 1, 2, 2, 3, 1, 2, 2, 1, 3, 2, 2, 3, 1, 2, 2, 1, 4, 2, 2, 2, 3, 2, 2, 3, 1, 2, 1, 3, 2, 4, 5, 2, 2, 1, 3, 2, 1, 2, 3, 1, 2, 2, 3, 2, 1, 2, 0, 0, 1, 4, 2, 2, 1, 2, 3, 1, 2, 1, 2, 3, 1, 2, 1, 1, 2, 3, 0, 1, 2, 1, 2, 1, 2, 4, 1, 2, 2, 2, 3, 1, 2, 1, 2, 3, 2, 0, 8, 3, 1, 2, 1, 4, 2, 1, 0, 2, 3, 1, 2, 2, 1, 3, 2, 2, 1, 2, 3, 1, 2, 0, 2, 1, 2, 4, 5, 1, 2, 1, 2, 1, 2, 1, 3, 1, 2, 1, 1, 2, 1, 2, 3, 1, 2, 1, 1, 2, 1, 8, 1, 1, 2, 1, 2, 1, 2, 2, 1, 3, 2, 1, 2, 3, 1, 2, 1, 2, 3, 2, 1, 2, 8]
		line_colors = ["#FF0000","#FF0000","#FF0000","#FF0000","#FF0000","#FF0000","#FF0000","#FF0000","#FF0000"]
		line_opacities = [1.00,0.50,1.00,0.50,1.00,0.50,1.00,0.50,1.00]
		line_width = [2,1,2,1,2,1,2,1,2]

		dic = {}
		enc_points_array = []
		enc_points_array.append(enc_points)
		dic['encoded_points'] = enc_points_array
		dic['linecolors'] = line_colors
		dic['lineopacities'] = line_opacities
		dic['linewidth'] = line_width
		dic['points'] = createPoints(points)

		with open(MapsFilesDir+'/data.json', 'w') as f:
			json.dump(dic, f, indent=1)

		# gmapmpr.js
		# comment.txtの変更で表示・非表示が変わるのでこのあと

		# make 距離 - 高度 graph
		print("Make distance - altitude graph")
		x = []
		y = []
		for pt in points:
			x.append(pt['cum'])
			y.append(pt['alt'])
		writeGraph(x, y, filename='dist-alt.png', ylim=[alt_min, alt_max])
		#Note: altitudeの最低高度が必ずzeroになってしまうため

		# make 時刻 - 高度 graph
		print("Make time - altitude graph")
		x = []
		y = []
		for pt in points:
			x.append(pt['time'])
			y.append(pt['alt'])
		# print(x)
		writeTimeGraph(x, y, AXIS_ALT, ylim=[alt_min, alt_max], filename='time-alt.png')
		#Note: altitudeの最低高度が必ずzeroになってしまうため

		# make 時刻 - 平均速度 graph
		print("Make time - velocity graph")
		x = []
		y = []
		ymax = 0
		for pt in avg_points:
			x.append(pt['time'])
			y.append(pt['vel'])
			# print(pt['time'])
			# 擬似的に棒グラフにするため
			x.append(pt['time'] + datetime.timedelta(seconds=VELOCITY_UNIT*60-1))
			vel = pt['vel']
			y.append(vel)
			if ymax < vel:
				ymax = vel
		writeTimeGraph(x, y, AXIS_VEL, ymax=ymax, filename='time-speed.png')

		# make 時刻 - 距離 graph
		print("Make time - distance graph")
		x = []
		y = []
		ymax = 0
		for pt in points:
			x.append(pt['time'])
			cum = pt['cum']
			y.append(cum)
			if ymax < cum:
				ymax = cum
		writeTimeGraph(x, y, AXIS_DIST, ymax=ymax, filename='time-dst.png')

	else:
		stat = loadStat()

	if readWriteEnvGraph():
		EnvGraphHidden = ''

	if not args.load_parsed_data:
		# sort photos for initial process when without comment.txt
		if len(RPhotoNames) == 0:
			sortPhotosByTime(photos)

		if points:
			# 距離、速度を求める
			print("Find address in photographs")
			for photo in photos:
				if 'time' in photo:
					dt = photo['time']
					if not 'lat' in photo or not 'alt' in photo:
						pos, dir = getPosFromTime(points, dt)
						if not 'lat' in photo:
							photo['lat'] = pos['lat']
							photo['lng'] = pos['lng']
						if not 'alt' in photo and dir==1:
							photo['alt'] = pos['alt']
							photo['cum'] = pos['cum']
					if not args.no_addr:
						addr = getAddrFromPos(photo['lat'], photo['lng'])
						photo['addr'] = addr

			print("Find address in markers")
			# markerは必ず座標があるのでphotoとは扱いがちょっと異なる
			for marker in markers:
				if 'time' in marker:
					if not 'alt' in marker:
						dt = marker['time']
						pos, dir = getPosFromTime(points, dt)	# timeではなくlng,latでも求められるが。。
						if dir == 1:
							marker['alt'] = pos['alt']
							marker['cum'] = pos['cum']
				if not args.no_addr:
					# print(f'{marker['lat']=}, {marker['lng']=}')
					addr = getAddrFromPos(marker['lat'], marker['lng'])
					marker['addr'] = addr

		print("Saving in json...")
		saveInputData(points, photos, markers)

	region = getRegionFromPoints(points, photos)

	if points:
		# gmapmpr.js
		writeGmapMpr(photos, markers, region)

	if not args.only_comment and make_thumb:
		makeThumbnails(photos)

	photos = addDummyPhoto(photos)
	addAbstractPhotos()

	vars = buildVarsForHtml(points, photos, stat)
	saveVars(vars)
	if points:
		index_tpl = 'index-tpl.html'
		if stat['car']:
			index_tpl = 'index-car-tpl.html'
	else:
		index_tpl = 'index-notrack-tpl.html'
	writeTemplateFile(SCRIPT_PATH + '/' + index_tpl, 'index.html', vars)
	# writeTemplateFile(SCRIPT_PATH + '/sub-tpl.htm', MapsFilesDir+'/sub.htm', vars)
	writeTemplateFile(SCRIPT_PATH + '/style1-tpl.css', MapsFilesDir+'/style1.css', vars)
	writeTemplateFile(SCRIPT_PATH + '/style2-tpl.css', MapsFilesDir+'/style2.css', vars)

	if len(MarkedFiles) > 0:
		with open(PhotoMarksFile, 'w') as f:
			for file in MarkedFiles:
				print(file)
				f.write(file + '\n')
		with open(PhotoMarkCommentFile, 'w', encoding='utf-8') as f:
			for base in MarkedFileComments.keys():
				comment = MarkedFileComments[base]
				f.write(base + '.jpg\n' + comment + '\n')

	if args.ok:
		with open('ok', 'w') as f:
			f.write("OK")

def readProperties():
	global Properties, GOOGLE_MAPS_KEY, BODY_WEIGHT, LOAD_WEIGHT

	Properties['AbstractTitle'] = AbstractTitle

	for line in open(SCRIPT_PATH + '/' + PropertiesFile):
		line = line.rstrip('\n')
		param = line.split('=')
		if len(param) >= 2:
			name = param[0]
			value = param[1]
			if name == 'GOOGLE_MAPS_KEY':
				GOOGLE_MAPS_KEY = value
			elif name == 'DEFAULT_BODY_WEIGHT':
				BODY_WEIGHT = float(value)
			elif name == 'DEFAULT_LOAD_WEIGHT':
				LOAD_WEIGHT = float(value)
			else:
				Properties[name] = value

def updatePoints():
	print("Updating points.txt ...")
	import urllib.request
	url = 'http://pdic.sakura.ne.jp/cgi-bin/sharepoint/top.cgi?action=download'
	savename = SCRIPT_PATH + '/points.txt'
	urllib.request.urlretrieve(url, savename)

def saveInputData(points, photos, markers):
	with open(MapsFilesDir+'/points.json', 'w') as f:
		json.dump(points, f, indent=1, default=json_serial)
	with open(MapsFilesDir+'/photos.json', 'w') as f:
		json.dump(photos, f, indent=1, default=json_serial)
	with open(MapsFilesDir+'/markers.json', 'w') as f:
		json.dump(markers, f, indent=1, default=json_serial)

def loadInputData():
	with open(MapsFilesDir+'/points.json', 'r') as f:
		points = json.load(f)
	with open(MapsFilesDir+'/photos.json', 'r') as f:
		photos = json.load(f)
	markers = []
	if os.path.exists(MapsFilesDir+'/markers.json'):
		with open(MapsFilesDir+'/markers.json', 'r') as f:
			markers = json.load(f)

	# str -> DataTime conversion
	for pt in points:
		pt['time'] = str2datetime(pt['time'])
	for photo in photos:
		if 'time' in photo:
			photo['time'] = str2datetime(photo['time'])
	for marker in markers:
			if 'time' in marker:
				marker['time'] = str2datetime(marker['time'])

	return points, photos, markers

def saveStat(stat):
	with open(MapsFilesDir+'/stat.json', 'w') as f:
		json.dump(stat, f, indent=1)

def loadStat():
	file = MapsFilesDir+'/stat.json'
	if not os.path.exists(file):
		stat = {}
		return stat
	with open(file) as f:
		stat = json.load(f)
	return stat

def saveVars(vars):
	with open(MapsFilesDir+'/vars.json', 'w') as f:
		json.dump(vars, f, indent=1)

# 山座標データの読み込み
def loadMountPoints():
	mpoints, names = loadMountFile(MountPointsFile)
	mpoints, _ = loadMountFile(PointsFile, mpoints, names)
	# release: pts[0]['names'] = {}
	return mpoints

def loadMountFile(file, pts=None, names=None):
	if not pts:
		pts = []
	if not names:
		names = {}
	for line in open(SCRIPT_PATH + '/' + file, encoding='utf8'):
		if line[0] == '#':
			continue
		items = line.split('\t')
		if len(items) >= 4:
			name = items[0]
			pt = {}
			if False:	# 重複チェックあり
				if name in names:
					# 重複、後データ優先
					print("重複: ", name)
					i = names[name]
				else:
					i = len(pts)-1
					names[name] = i
					pt['name'] = name
					pts.append(pt)
				pts[i]['alt'] = float(items[1])
				pts[i]['lat'] = float(items[2])
				pts[i]['lng'] = float(items[3])
			else:
				pt['name'] = name
				pt['alt'] = float(items[1])
				pt['lat'] = float(items[2])
				pt['lng'] = float(items[3])
				pts.append(pt)
	return pts, names

def json_serial(obj):
    # 日付型の場合には、文字列に変換
    if hasattr(obj, "isoformat"):
        return obj.isoformat()
    raise TypeError ("Type %s not serializable" % type(obj))

def get_jpg_info(file):
	import PIL.ExifTags as ExifTags
	try:
		img = Image.open(file)
	except:
		print("open error:", file)
		return None

	exif = img._getexif()

	usercomment = None
	datetimeorg = None
	orientation = 0
	gpsdate = None
	gpstime = None
	lat = None
	lng = None
	alt = None

	if not exit or not hasattr(exif, "items"):
		print("Warning: no exif info: ", file)
		return datetimeorg, orientation, lat, lng, alt, ''

	# attrs = {}
	# for id,val in exif.items():
	# 	tg = TAGS.get(id)
	# 	attrs[tg] = val
	# 	print(tg, val)

	UserComment = 0x9286
	if UserComment in exif:
		b_str = exif[UserComment]
		if b_str.startswith(b'UNICODE\0'):	# Exif Editorが悪いのかどうか不明だけど、余計な文字が先頭に入っている？
			b_str = b_str[8:]
			# 動作未確認
			if b_str.startswith(b'\xEE\xBB\xBF'):
				usercomment = b_str.decode('utf8', 'replace')
			else:
				usercomment = b_str.decode('utf16', 'replace')
		else:
			usercomment = b_str.decode()

	Orientation = 274
	if Orientation in exif:
		orientation = exif[Orientation]

	def toFloat(s):
		return float(s[0][0]) / float(s[0][1]) + float(s[1][0]) / float(s[1][1]) / 60 + (float(s[2][0])/float(s[2][1])) /3600

	def toTime(s):
		hour = int(s[0][0]/s[0][1])
		minute = int(s[1][0]/s[1][1])
		second = int(s[2][0]/s[2][1])
		msec = int((s[2][0]/s[2][1] - second) * 1000000)
		# print(hour, minute, second, msec)
		return datetime.datetime(2000, 1, 1, hour=hour, minute=minute, second=second, microsecond=msec)

	def toDate(s):
		m = re.match(r'(\d+):(\d+):(\d+)', s)
		if m:
			# print(s)
			return datetime.datetime(int(m.group(1)), int(m.group(2)), int(m.group(3)))
		return None

	GPSInfo = 34853
	if GPSInfo in exif:
		gtag = exif[GPSInfo]
		INX_LAT_REF = 1
		INX_LAT = 2
		INX_LNG_REF = 3
		INX_LNG = 4
		INX_ALT_REF = 5
		INX_ALT = 6
		INX_GPSTIME = 7
		INX_GPSDATE = 29
		if type(gtag) == dict:
			if INX_LAT in gtag and INX_LNG in gtag:
				# print(gtag)
				lat = toFloat(gtag[INX_LAT])
				lng = toFloat(gtag[INX_LNG])
				if gtag[INX_LAT_REF] != 'N':
					lat = -lat
				if gtag[INX_LNG_REF] != 'E':
					lng = -lng
			if INX_ALT in gtag:
				alt = float(gtag[INX_ALT][0]) / float(gtag[INX_ALT][1])
				if gtag[INX_ALT_REF] != b'\x00':	# below sea level
					alt = -alt
			if INX_GPSTIME in gtag:
				gpstime = toTime(gtag[INX_GPSTIME])
			if INX_GPSDATE in gtag:
				gpsdate = toDate(gtag[INX_GPSDATE])

		# else:
		# 	print(gps_tags)	# 31714 って何？

	DateTimeOriginal = 36867
	if DateTimeOriginal in exif:
		dstr = exif[DateTimeOriginal]
		# print(dstr)
		t = datetime.datetime.strptime(dstr, '%Y:%m:%d %H:%M:%S')
		if gpstime:
			t = t.replace(hour=gpstime.hour, minute=gpstime.minute, second=gpstime.second, microsecond=gpstime.microsecond, tzinfo=datetime.timezone.utc)
		if gpsdate:
			t = t.replace(year=gpsdate.year, month=gpsdate.month, day=gpsdate.day)
		datetimeorg = t.astimezone(datetime.timezone(datetime.timedelta()))	# as UTC

	return datetimeorg, orientation, lat, lng, alt, usercomment

def getPosFromTime(points, dt):
	assert(len(points)>0)
	if dt < points[0]['time']:
		# print("Found1:", dt, points[0]['time'])
		return points[0], 0	# pointの範囲の外（前）
	if dt == points[0]['time']:
		return points[0], 1
	prev_pt = points[0]
	for pt in points:
		# print(dt, pt['time'])
		if dt < pt['time']:
			if Verbose:
				print("Found2: ", dt, pt['lat'], pt['lng'])
			if False:
				return prev_pt
			else:	# 補完処理
				td = (dt - prev_pt['time']).seconds / (pt['time'] - prev_pt['time']).seconds
				ret_pt = prev_pt
				ret_pt['lat'] = (pt['lat'] - prev_pt['lat']) * td + prev_pt['lat']
				ret_pt['lng'] = (pt['lng'] - prev_pt['lng']) * td + prev_pt['lng']
				ret_pt['alt'] = (pt['alt'] - prev_pt['alt']) * td + prev_pt['alt']
				ret_pt['cum'] = (pt['cum'] - prev_pt['cum']) * td + prev_pt['cum']
				# print(ret_pt)
				return ret_pt, 1
		prev_pt = pt
	# print("Not found: ", dt)
	return prev_pt, 2

def getRegionFromPoints(points, photos):
	def getRegion(pts, rc):
		pt = pts[0]
		for pt in pts:
			if 'lat' in pt:
				lng = pt['lng']
				lat = pt['lat']
				if not rc:
					rc = [lng, lat, lng, lat]
					continue
				if rc[0] > lng:
					rc[0] = lng
				elif rc[2] < lng:
					rc[2] = lng
				if rc[1] > lat:
					rc[1] = lat
				elif rc[3] < lat:
					rc[3] = lat
		return rc

	rc = None

	if len(points) > 0:
		rc = getRegion(points, rc)

	if len(photos) > 0:
		rc = getRegion(photos, rc)

	if not rc:
		return None

	# inflate
	w = rc[2] - rc[0]
	h = rc[3] - rc[1]
	ratio = REGION_INFLATE_RATIO
	rc = [rc[0] - w*ratio, rc[1] - h*ratio, rc[2] + w*ratio, rc[3] + h*ratio]

	return rc

def getAddrFromPos(lat, long):
	addr = ''

	gmaps = googlemaps.Client(key=GOOGLE_MAPS_KEY)
	arg = (lat, long)
	# print(arg)
	# arg = (35.675888, 139.744858)	# 国会議事堂
	# arg = (35.43749, 139.36202)	# 自宅
	results = gmaps.reverse_geocode(arg, language=GOOGLE_MAPS_LANG)
	for result in results:
		types = result['types']
		if 'sublocality_level_3' in types:
			# print('types: ', types)
			# 国名、郵便番号はいらないので再構築
			for item in result['address_components']:
				types = item['types']
				# print(types)
				if 'political' in types:
					if 'sublocality_level_3' in types or 'sublocality_level_2' in types or 'locality' in types or 'administrative_area_level_1' in types:
						addr = item['long_name'] + addr
						# addr = item['short_name'] + addr

	if not addr:
		# sublocality_level_3がない場合
		for result in results:
			if 'address_components' in result:
				addrs = {}
				for item in result['address_components']:
					types = item['types']
					if 'sublocality' in types:
						addrs['sublocal'] = item['long_name']
					if 'locality' in types:
						addrs['local'] = item['long_name']
					if 'administrative_area_level_1' in types:
						addrs['pref'] = item['long_name']
				if len(addrs) == 3:
					addr = addrs['pref'] + addrs['local'] + addrs['sublocal']
					break

	return addr

# point間の距離・速度を求める
def calcDistanceVelocity(points):
	prev = None
	dist_cum = 0	# 累積距離
	alt_min = 1000000
	alt_max = -1000000
	for i, pt in enumerate(points):
		if prev and prev['time']:
			pt['dst'] = geodesic((prev['lat'], prev['lng']), (pt['lat'], pt['lng'])).km
			delta = pt['time'] - prev['time']
			if delta.seconds == 0:
				pt['vel'] = points[i-1]['vel']
			else:
				pt['vel'] = pt['dst'] / delta.seconds * 3600
			dist_cum += pt['dst']
			pt['cum'] = dist_cum
		else:
			pt['dst'] = 0
			pt['vel'] = 0
			pt['cum'] = 0
		prev = pt
		pt['rlat'] = round(pt['lat'], '0.00000')
		pt['rlng'] = round(pt['lng'], '0.00000')
		if alt_min > pt['alt']:
			alt_min = pt['alt']
		elif alt_max < pt['alt']:
			alt_max = pt['alt']

	return dist_cum, alt_min, alt_max

# 実移動距離（休憩時間）を求める
# YAMAPからdownloadしたgpxだと休憩時間のdataが省略されるらしく、正しく認識できないときがある（かもしれない）
def getRests(points):
	REST_DEBUG = False
	rests = []
	sum_rest_time = 0
	i = 0
	while i < len(points):
		pt = points[i]
		if pt['dst']:
			# ptを基準にした距離ではなくて、pt以降の平均（または全体からの）座標を基準にした距離で休憩を判断
			sum_lat = pt['lat']
			sum_lng = pt['lng']
			sum_count = 1
			for j in range(i+1, len(points)):
				pt2 = points[j]

				# center座標/距離を求める
				center_lat = sum_lat / sum_count
				center_lng = sum_lng / sum_count
				dst = geodesic((center_lat, center_lng), (pt2['lat'], pt2['lng'])).km
				sum_lat += pt2['lat']
				sum_lng += pt2['lng']
				sum_count += 1

				if REST_DEBUG: print("i={}({}) dst={} time={}".format(i, pt['time'], dst*1000, pt2['time']))
				rest_continue = False
				if dst >= MaxRestArea / 1000:
					# 対象外
					# n分以内にptに戻ってくるか？
					for k in range(j+1, len(points)):
						pt3 = points[k]
						delta = pt3['time'] - pt2['time']
						if delta.seconds >= MaxReturnTime:
							# 時間が過ぎてしまった
							if REST_DEBUG: print("k={} out of time: {}".format(k, delta.seconds))
							break
						dst = geodesic((center_lat, center_lng), (pt3['lat'], pt3['lng'])).km
						if dst < MaxRestArea / 1000:
							# 戻ってきた
							if REST_DEBUG: print("Return to rest area")
							rest_continue = True
							break

					if rest_continue:
						continue	# 休憩継続

					if pt2['time'] < pt['time']:
						print("time order is not correct in gps log file(gpx/log).", pt['time'], pt2['time'], j, i)
						exit(1)

					delta = pt2['time'] - pt['time']
					if delta.seconds >= MinRestTime:
						if REST_DEBUG: print("■Found: i={} j={} found rest:{} delta={}".format(i, j, pt['time'], delta.seconds))
						rest = {'index':i, 'dur':delta.seconds, 'num':(j-i)}
						rests.append(rest)
						sum_rest_time += delta.seconds
						i = j - 1
						if REST_DEBUG: print("change i to ", i+1)
					else:
						if REST_DEBUG:
							print("Not found: {} delta={} i={} j={}".format(pt['time'], delta.seconds, i, j))
							for k in range(i+1, j+1+2):
								pt2 = points[k]
								dst = geodesic((pt['lat'], pt['lng']), (pt2['lat'], pt2['lng'])).km
								print(pt2['time'], dst*1000)
					break
		i += 1
	return rests, sum_rest_time

# peakの高低差を調べる
# restsを使って正確な移動時間の計算
def getPeaks(points, rests, min_elev):
	peaks = []

	last_min = 1000000
	last_min_index = 0
	min_found = False
	last_max = -1000000
	last_max_index = 0
	max_found = False
	total_up = 0
	total_dn = 0
	total_up_dst = 0
	total_dn_dst = 0
	sum_dst = 0			# up or downの累積距離
	sum_rest_time = 0	# up or down中の休憩時間[sec] それぞれのup or down時間から差し引くために使用
	rest_num = 0
	total_up_time = 0
	total_dn_time = 0
	last_peak = points[0]['alt']
	prev_time = points[0]['time']
	last_min_time = prev_time
	last_max_time = prev_time
	last_peak_time = prev_time
	peak_count = 0	# the number of top peaks

	for i, pt in enumerate(points):
		alt = pt['alt']
		tm = pt['time']
		dst = pt['dst']
		# print(int(alt))
		if alt > last_max:
			# 上り
			last_max = alt
			last_max_index = i
			last_max_time = tm
			# print("max diff=", int(alt-last_min))
			if min_found:
				last_min = alt
				last_min_index = i
				last_min_time = tm
			elif alt - last_min >= min_elev:
				if not min_found:
					# print("!!!!found min", last_min)
					min_found = True
					max_found = False
					peak = {'index': last_min_index, 'alt':last_min}
					peaks.append(peak)
					if last_min > last_peak:
						total_up += last_min - last_peak
						total_up_dst += sum_dst
						move_time, sum_rest_time = getMoveTime((last_min_time - last_peak_time).seconds, sum_rest_time)
						total_up_time += move_time
						# print("t1: {}min {:.2f}km {:.1f}m".format(move_time//60, sum_dst, last_min))
					else:
						total_dn += last_peak - last_min
						total_dn_dst += sum_dst
						move_time, sum_rest_time = getMoveTime((last_min_time - last_peak_time).seconds, sum_rest_time)
						total_dn_time += move_time
						# print("t2: {}min {}->{} {:.2f}km {:.1f}m rest={}sec".format(move_time//60, last_peak_time, last_min_time, sum_dst, last_min, sum_rest_time))
					# print("total_up_time={} dn={}".format(total_up_time, total_dn_time))
					sum_dst = 0
					last_peak = last_min
					last_peak_time = last_min_time
		elif alt < last_min:
			# 下り
			last_min = alt
			last_min_index = i
			last_min_time = tm
			# print("min diff=", int(last_max-alt))
			if max_found:
				last_max = alt
				last_max_index = i
				last_max_time = tm
			elif last_max - alt >= min_elev:
				if not max_found:
					# print("!!!!found max", last_max)
					max_found = True
					min_found = False
					peak = {'index': last_max_index, 'alt':last_max}
					peaks.append(peak)
					peak_count += 1
					if last_max > last_peak:
						# print("peak: {:.2f} {:.2f} {:.2f}".format(last_peak, last_max, last_max-last_peak))
						total_up += last_max - last_peak
						total_up_dst += sum_dst
						move_time, sum_rest_time = getMoveTime((last_max_time - last_peak_time).seconds, sum_rest_time)
						total_up_time += move_time
						# print("t3: {}min {:.2f}km {:.1f}m rest={}".format(move_time//60, sum_dst, last_max, sum_rest_time))
					else:
						total_dn += last_peak - last_max
						total_dn_dst += sum_dst
						move_time, sum_rest_time = getMoveTime((last_max_time - last_peak_time).seconds, sum_rest_time)
						total_dn_time += move_time
						# print("t4: {}min {:.2f}km {:.1f}m".format(move_time//60, sum_dst, last_max))
					# print("total_up_time={} dn={}".format(total_up_time, total_dn_time))
					sum_dst = 0
					last_peak = last_max
					last_peak_time = last_max_time

		MinRestTime = MIN_REST_TIME
		num = isRest(rests, i, MinRestTime)
		if num:
			if rest_num == 0:
				rest_num = num
				if i+1 < len(points):
					sum_rest_time += (points[i+1]['time'] - tm).seconds
					# print("rest time added[{}]={}".format(i, sum_rest_time))
			rest_num -= 1
		else:
			assert(rest_num==0)
			sum_dst += dst
		prev_time = tm

	alt = points[-1]['alt']
	tm = points[-1]['time']
	if alt > last_peak:
		total_up += alt - last_peak
		total_up_dst += sum_dst
		move_time, sum_rest_time = getMoveTime((tm - last_peak_time).seconds, sum_rest_time)
		total_up_time += move_time
	else:
		total_dn += last_peak - alt
		total_dn_dst += sum_dst
		move_time, sum_rest_time = getMoveTime((tm - last_peak_time).seconds, sum_rest_time)
		total_dn_time += move_time

	return peaks, total_up, total_dn, peak_count, total_up_dst, total_dn_dst, total_up_time, total_dn_time

def getMoveTime(difftime, rest_time):
	if difftime < rest_time:
		return 0, rest_time - difftime
	else:
		return difftime - rest_time, 0

def isRest(rests, index, minsec):
	if not rests:
		return 0
	for i, rest in enumerate(rests):
		if rest['dur'] >= minsec:
			# 対象の休憩時間
			rindex = rest['index']
			# indexはこの休憩時間中か？
			if index >= rindex and index < rindex + rest['num']:
				return rest['num']
	return 0


def getMounts(points):
	FindNearest = True	# 最も近づいたときを到着時刻 or 200m以内に最初に入った時を到着時刻

	MovingSearchArea = 5	# [km]
	NearDistance = MountArriveDistance/1000	# [km]
	NearestDistance = 0.02	# [km] これ以下になった到着確定
	NotNearDistance = 0.5	# [km]

	global MPoints
	MPoints = loadMountPoints()
	mounts = []
	npoints = []
	last_mts = {}
	pvpt = None
	for i, pt in enumerate(points):
		if not pvpt or getDistance(pvpt, pt) > MovingSearchArea:
			pvpt = pt
			npoints = getSubMPoints(pt)
		mts = findPointsInPts(pt, npoints, NearDistance)
		if len(mts) > 0:
			for mt in mts:
				name = mt['name']
				if name in last_mts:
					if FindNearest:
						# 一番近いものを残す
						index = last_mts[name]['index']
						dist = getDistance(pt, mounts[index])
						if dist < last_mts[name]['dist'] and last_mts[name]['dist'] > NearestDistance:
							# update point index
							last_mts[name]['dist'] = dist
							mounts[index]['index'] = i	# index of points
							# print("Update: [{}] dist={:.2f} {}".format(i, dist, pt['time']))
				else:	# new mountaine
					# print("Enter: ", name)
					last_mts[name] = mt.copy()
					mounts.append(mt.copy())
					index = len(mounts)-1
					mounts[index]['index'] = i	# index of points
					last_mts[name]['index'] = index
					last_mts[name]['dist'] = getDistance(pt, mt)
		else:
			for name in list(last_mts.keys()):
				dist = getDistance(pt, last_mts[name])
				if dist > NotNearDistance:
					del last_mts[name]
	return mounts

# ptから半径10kmの山座標データを取得
def getSubMPoints(cpt):
	return findPointsInPts(cpt, MPoints, 10)

# ptsの中で、cptから指定距離内にcptがあるpointを返す
def findPointsInPts(cpt, pts, distance):
	rpts = []
	lat = cpt['lat']
	lng = cpt['lng']
	for pt in pts:
		dst = geodesic((lat, lng), (pt['lat'], pt['lng'])).km
		if dst < distance:
			rpts.append(pt)
	return rpts

def getDistance(pt1, pt2):
	return geodesic((pt1['lat'], pt1['lng']), (pt2['lat'], pt2['lng'])).km

def read_gpx(file):
	import gpxpy
	import gpxpy.gpx
	gpx_file = open(file, 'r', encoding='utf8')
	gpx = gpxpy.parse(gpx_file)

	points = []

	for track in gpx.tracks:
		for segment in track.segments:
			for point in segment.points:
				if not point.time:
					continue	# 時刻の無い位置情報は破棄？
				p = {}
				p['lat'] = point.latitude
				p['lng'] = point.longitude
				p['alt'] = point.elevation
				p['time'] = point.time	# UTC
				points.append(p)

	points.sort(key=lambda x: x['time'])
	print("Read {} points.".format(len(points)))

	return points

def parse_gpx(file):
	import gpxpy
	import gpxpy.gpx
	gpx_file = open(file, 'r')

	gpx = gpxpy.parse(gpx_file)

	print("-- tracks --")
	for track in gpx.tracks:
		print("-- start track: segs={}".format(len(track.segments)))
		for segment in track.segments:
			print("-- start segment: num={}".format(len(segment.points)))
			for point in segment.points:
				print('Point at ({0},{1}) -> {2} {3}'.format(point.latitude, point.longitude, point.elevation, point.time))

	print("-- waitpoints --")
	for waypoint in gpx.waypoints:
		print('waypoint {0} -> ({1},{2})'.format(waypoint.name, waypoint.latitude, waypoint.longitude))

	print("-- routes --")
	for route in gpx.routes:
		print('Route:')
		for point in route.points:
			print('Point at ({0},{1}) -> {2}'.format(point.latitude, point.longitude, point.elevation))

def read_nmea(file, date):
	import pynmea2
	points = []
	prev_time = None
	for line in open(file):
		if line.startswith('@'):
			continue
		msg = pynmea2.parse(line)
		# print(repr(msg))
		if msg and msg.sentence_type == 'GGA':
			p = {}
			p['lat'] = msg.latitude
			p['lng'] = msg.longitude
			p['alt'] = msg.altitude
			dt = datetime.datetime(year=date.year, month=date.month, day=date.day, hour=msg.timestamp.hour, minute=msg.timestamp.minute, second=msg.timestamp.second, tzinfo=datetime.timezone.utc)
			if msg.timestamp.hour + LOCALTIME_DIFF >= 24:
				# 一つ前の日付にする必要あり
				dt -= datetime.timedelta(days=1)
			p['time'] = dt
			# print(p['time'])
			if prev_time == p['time']:
				# NMEAには同じ時刻データが入るときがある？
				continue
			prev_time = p['time']
			points.append(p)

	print("Read {} points.".format(len(points)))

	return points

def read_kml(file):
	import xml.etree.ElementTree as ET
	
	points = []
	
	tree = ET.parse(file)
	root = tree.getroot()
	
	for placemark in root.iter('{http://www.opengis.net/kml/2.2}Placemark'):
		coordinates = placemark.find('{http://www.opengis.net/kml/2.2}Point/{http://www.opengis.net/kml/2.2}coordinates').text
		name = placemark.find('{http://www.opengis.net/kml/2.2}name')
		if name is not None:
			name = name.text
		description = placemark.find('{http://www.opengis.net/kml/2.2}description')
		if description is not None:
			description = description.text
		timestamp = placemark.find('{http://www.opengis.net/kml/2.2}TimeStamp/{http://www.opengis.net/kml/2.2}when')
		if timestamp is not None:
			timestamp = datetime.datetime.strptime(timestamp.text, "%Y-%m-%dT%H:%M:%SZ").replace(tzinfo=datetime.timezone.utc)
		color = placemark.find('{http://www.opengis.net/kml/2.2}Style/{http://www.opengis.net/kml/2.2}IconStyle/{http://www.opengis.net/kml/2.2}color')
		if color is not None:
			color = color.text
	
		# print(f'{coordinates} {name=} {description=} {timestamp=} {color=}')
		lng, lat, _ = coordinates.split(',')
		
		point = {
			'name': name,
			'description': description,
			'lat': float(lat),
			'lng': float(lng),
			'time': timestamp,
			'color': color		# ffrrggbb
		}
		
		points.append(point)
	
	return points

def parse_jpg(file):
	try:
		img = Image.open(file)
	except:
		print("open error:", file)
		return

	exif = img._getexif()
	try:
		for id,val in exif.items():
			tg = TAGS.get(id,id)
			if tg == 'MakerNote':
				continue	# 長いので
			print(tg, val)
	except AttributeError:
		print("AttributeError")
	
	img.close()

def encodePoints(points):
	ptp = []
	for pt in points:
		ptp.append((pt['lat'], pt['lng']))

	return polyline.encode(ptp)

def decodePoints(enc_points):
	return polyline.decode(enc_points)

def decodeLevel(str):
	ret_levels = []

	for c in str:
		c = ord(c) - 63
		ret_levels.append(c)

	return ret_levels

def createPoints(points):
	ret = []
	for pt in points:
		dts = pt['time'].strftime('%Y/%m/%d %H:%M')
		dst = round(pt['vel'], '0.00')
		ret.append([pt['rlat'], pt['rlng'], dst, dts])

	return ret

def makeWayPoints(photos):
	s = ''
	for i, photo in enumerate(photos):
		if not 'time' in photo:
			continue
		name = photo['name']

		# 非表示対応 - Firefoxだとcacheされる（？）ので非表示にならない（その逆もあり）
		if name in RTitle:
			title = RTitle[name]
			if title.startswith('*'):
				continue	# 非表示

		dt = photo['time'] + datetime.timedelta(hours=LOCALTIME_DIFF)
		ts = dt.strftime("%Y:%m:%d %H:%M:%S")
		lng = lngToDMS(photo['lng'])
		lat = latToDMS(photo['lat'])
		width = photo['thw']
		height = photo['thh']
		if 'alt' in photo:
			alt = photo['alt']
		else:
			alt = 0
		alt = round(alt, 1)
		addr = photo['addr'] if 'addr' in photo else ''
		s += '	addWayPoints(map, 1,{},{},"<b><a href=\\""+relativepathtomainhtml+"#photoID{}\\">{}</a></b><br /><a href=\\""+"{}.jpg\\"  target=\\"_blank\\"><img src=\\""+relativepathtosubfolder+"{}-thumb.jpg\\" border=\\"0\\" width=\\"{}\\" height=\\"{}\\" /></a><br /><small>{}<br />{}<br />{}<br />{}m<br />{}<hr /></small>");\n'.format(photo['lng'], photo['lat'], i+1, name, name, name, width, height, ts, lng, lat, alt, addr)
	return s

# markerのためのwaypoint作成
def makeWayPointsMaker(markers):
	s = ''
	for i, marker in enumerate(markers):
		name = marker['name']
		description = marker['description']
		if not description: description = ''

		ts = ''
		if 'time' in marker:
			dt = marker['time'] + datetime.timedelta(hours=LOCALTIME_DIFF)
			ts = dt.strftime("%Y:%m:%d %H:%M:%S")
		lng = lngToDMS(marker['lng'])
		lat = latToDMS(marker['lat'])
		alt = marker['alt'] if 'alt' in marker else 0
		alt = round(alt, 1)
		addr = marker['addr'] if 'addr' in marker else ''
		color = marker['color']
		nType = colorToType(color)
		s += f'	addWayPoints(map, {nType},{marker['lng']},{marker['lat']},"<b>{name}</b><br />{description}<br /><small>{ts}<br />{lng}<br />{lat}<br />{alt}m<br />{addr}<hr /></small>");\n'
		# print(f'{name=} {description=} {lng=} {lat=} {alt=} {addr=} {color=} {nType=}')
	return s

def colorToType(color):
	# gmapmgr-tpl.js - addWayPoints - nType参照

	if color is not None:
		color = color & 0xFFFFFF

	if color == 0xFF0000:
		return 25	# red
	elif color == 0x00FF00:
		return 24	# green
	elif color == 0xFFFF00:
		return 21	# yellow
	elif color == 0xFF00FF:
		return 20	# magenta/purple
	elif color == 0xFFA500:
		return 27	# orange
	elif color == 0x00FFFF:
		return 13	# cyan/lightblue	#TODO:
	elif color == 0xFFC0CB:
		return 16	# pink				#TODO:
	elif color == 0x000000:
		return 26	# black
	else:	# else or blue(0x0000FF)
		return 22

# 小数点形式(DEG形式)->DMS形式変換
def lngToDMS(pos):
	ew = 'E' if pos > 0 else 'W'
	d = int(pos)
	m = int((pos - d) * 60)
	s = ((pos - d) - (m/60)) * 3600
	s = round(s, '0.00')
	return '{}{}°{}’{}”'.format(ew, d, m, s)

# 小数点形式(DEG形式)->DMS形式変換
def latToDMS(pos):
	ns = 'N' if pos > 0 else 'S'
	d = int(pos)
	m = int((pos - d) * 60)
	s = ((pos - d) - (m/60)) * 3600
	s = round(s, '0.00')
	return '{}{}°{}’{}”'.format(ns, d, m, s)

#
# Google Maps functions
#
def writeGmapMpr(photos, markers, region):
	# make waypoints for google map
	waypoints_s = makeWayPoints(photos)
	waypoints_s += makeWayPointsMaker(markers)
	# print(waypoints_s)

	vars = buildBasicVars()
	vars['NumOfPhotos'] = len(photos)
	vars['WayPoints'] = waypoints_s
	vars['RegionLeft'] = region[0]
	vars['RegionTop'] = region[1]
	vars['RegionRight'] = region[2]
	vars['RegionBottom'] = region[3]
	writeTemplateFile(SCRIPT_PATH + '/gmapmgr-tpl.js', MapsFilesDir+'/gmapmgr.js', vars)

#
# Graph functions
#

import pandas as pd
import matplotlib as mpl
import matplotlib.pyplot as plt
from matplotlib.dates import date2num
from matplotlib.dates import DateFormatter
import matplotlib.dates as mdates
import numpy as np
# import seaborn as sns

def writeGraph(x, y, filename=None, show=False, ylim=None):
	fig = plt.figure(figsize=(8,4), dpi=80)
	axes = fig.add_subplot(111)
	axes.yaxis.set_minor_locator(mpl.ticker.MultipleLocator(50))
	if ylim:
		plt.ylim(ylim[0], ylim[1])

	plt.fill_between(x, y, 0, facecolor='green', interpolate=False, alpha=0.3)
	# plt.fill_between(x[1:], y_returns[1:], 0, where=y_returns[1:] >= 0, facecolor='green', interpolate=False, alpha=0.3)
	# plt.fill_between(x[1:], y_returns[1:], 0, where=y_returns[1:] <= 0, facecolor='red', interpolate=True, alpha=0.7)

	# xtickvals = [str(m)[:3].upper()+"-"+str(y) for y,m in zip(df.date.dt.year, df.date.dt.month_name())]
	# plt.gca().set_xticks(x[::6])
	# plt.gca().set_xticklabels(xtickvals[::6], rotation=90, fontdict={'horizontalalignment': 'center', 'verticalalignment': 'center_baseline'})
	plt.title("Distance - Altitude", fontsize=16)
	plt.ylabel('Altitude [m]')
	plt.xlabel("Distance [km]")
	if show:
		plt.show()
	if filename:
		fig.savefig(MapsFilesDir+"/"+filename)

# xがdatetime型のgraph
AXIS_ALT = 0
AXIS_VEL = 1
AXIS_DIST = 2
def writeTimeGraph(x, y, y_axis, ylim=None, ymax=0, filename=None, show=False):
	x = toLocalTimeAbs(date2num(x))

	fig = plt.figure(figsize=(8,4), dpi=80)
	axes = fig.add_subplot(111)
	axes.xaxis.set_major_formatter(DateFormatter('%H:%M'))
	# axes.xaxis.set_major_locator(mdates.HourLocator(interval=1))
	# axes.xaxis.set_minor_locator(mdates.MinuteLocator(interval=10))
	axes.xaxis.set_major_locator(mdates.HourLocator(byhour=range(0, 24, 1), tz=None))
	axes.xaxis.set_minor_locator(mdates.MinuteLocator(byminute=range(0, 60, 30), tz=None))
	if ylim:
		plt.ylim(ylim[0], ylim[1])

	if y_axis == AXIS_ALT:
		ylabel = 'Altitude'
		yunit = '[m]'
		axes.yaxis.set_minor_locator(mpl.ticker.MultipleLocator(50))
	elif y_axis == AXIS_VEL:
		ylabel = 'Velocity'
		yunit = '[km/h]'
		if ymax < 200:
			# 時速200km/h未満のみ
			axes.yaxis.set_minor_locator(mpl.ticker.MultipleLocator(5))
	else:	# AXIS_DIST
		ylabel = 'Distance'
		yunit = '[km]'
		if ymax < 2000:
			# 移動距離が2,000km未満のみ
			axes.yaxis.set_minor_locator(mpl.ticker.MultipleLocator(10))

	plt.fill_between(x, y, 0, facecolor='green', interpolate=False, alpha=0.3)

	plt.title("Time - "+ylabel, fontsize=16)
	plt.ylabel(ylabel+' '+yunit)
	plt.xlabel('Time')
	if show:
		plt.show()
	if filename:
		fig.savefig(MapsFilesDir+"/"+filename)

def readWriteEnvGraph():
	# 気圧データの読み込み
	pressure_times = []
	pressure_values = []
	files = glob.glob('*.csv')
	for file in files:
		# 気圧データのファイル名は 20250408-130124.csv を前提
		if not re.match(r'\d{8}-\d{6}\.csv', file):
			continue
		print(f"Reading...{file}")
		times, values = read_csv(file, 1, False)
		pressure_times.extend(times)
		pressure_values.extend(values)
		pressure_values = list(map(float, pressure_values))

	# 温度データの読み込み
	temperature_times = []
	temperature_values = []
	for file in files:
		# temphumid.csv
		if not re.match(r'temphumid\.csv', file):
			continue
		print(f"Reading...{file}")
		times, values = read_csv(file, 1, True)
		temperature_times.extend(times)
		temperature_values.extend(values)
		temperature_values = list(map(float, temperature_values))

	# 湿度データの読み込み
	humidity_times = []
	humidity_values = []
	for file in files:
		# temphumid.csv
		if not re.match(r'temphumid\.csv', file):
			continue
		print(f"Reading...{file}")
		times, values = read_csv(file, 2, True)
		humidity_times.extend(times)
		humidity_values.extend(values)
		humidity_values = list(map(float, humidity_values))

	retval = False

	# 気圧、温度、湿度のグラフ作成
	if False:
		# 気圧・温度・湿度を同時表示
		if len(pressure_times) > 0 or len(temperature_times) > 0 or len(humidity_times) > 0:
			print("Make time - pressure/temperature/humidity graph")
			writeEnvGraph(pressure_times, pressure_values, temperature_times, temperature_values, humidity_times, humidity_values, filename='envgraph.png')
			global EnvGraphAllHidden
			EnvGraphAllHidden = ''
			return True
	else:
		if len(pressure_times) > 0:
			print("Make time - pressure graph")
			writeEnvGraphPressure(pressure_times, pressure_values, filename='envgraph-press.png')
			global EnvGraphPressHidden
			EnvGraphPressHidden = ''
			retval = True
		if len(temperature_times) > 0 or len(humidity_times) > 0:
			print("Make time - temperature/humidity graph")
			writeEnvGraphTempHumid(temperature_times, temperature_values, humidity_times, humidity_values, filename='envgraph-temp.png')
			global EnvGraphTempHidden
			EnvGraphTempHidden = ''
			retval = True
	return retval

# 気圧のgraphの作成
# X軸：時刻
# Y軸：気圧の表示。折れ線グラフを使用
def writeEnvGraphPressure(x_pressure, y_pressure, filename=None, show=False):

	#TODO: 日本語対応するため。どうする？
	# import matplotlib.font_manager as fm
	# font_path = "C:/Windows/Fonts/msgothic.ttc "  # Windows
	# # font_path = "/usr/share/fonts/truetype/ipaexg/ipaexg.ttf"	# Linux
	# jp_font = fm.FontProperties(fname=font_path)
	# plt.rcParams['font.family'] = jp_font.get_name()

	# 文字列をdatetimeに変換
	pressure_times = [datetime.datetime.strptime(t, "%Y-%m-%d %H:%M:%S") for t in x_pressure]

	# グラフの作成
	fig, ax1 = plt.subplots(figsize=(6.4, 4))

	# 気圧のプロット（y軸1）
	if len(y_pressure) > 0:
		ax1.plot(pressure_times, y_pressure, 'b-', label='PRESSURE[hPa]')
		ax1.set_ylabel("PRESSURE[hPa]", color='b')
		ax1.tick_params(axis='y', labelcolor='b')

	ax2 = None

	# 日時フォーマット
	ax1.xaxis.set_major_formatter(mdates.DateFormatter('%H:%M'))
	fig.autofmt_xdate()

	# 凡例の統合
	lines1 = []
	labels1 = []
	if len(y_pressure) > 0:
		lines1, labels1 = ax1.get_legend_handles_labels()
	ax1.legend(lines1, labels1, loc='upper left')

	# plt.title("気圧の変化")
	# plt.grid(True)
	plt.tight_layout()
	if show:
		plt.show()
	if filename:
		fig.savefig(MapsFilesDir+"/"+filename)

# 温度、湿度のgraphの作成
# X軸：時刻
# Y軸：温度、湿度の3つを同時に表示。折れ線グラフを使用
def writeEnvGraphTempHumid(x_temp, y_temp, x_humidity, y_humi, filename=None, show=False):

	#TODO: 日本語対応するため。どうする？
	# import matplotlib.font_manager as fm
	# font_path = "C:/Windows/Fonts/msgothic.ttc "  # Windows
	# # font_path = "/usr/share/fonts/truetype/ipaexg/ipaexg.ttf"	# Linux
	# jp_font = fm.FontProperties(fname=font_path)
	# plt.rcParams['font.family'] = jp_font.get_name()

	# 文字列をdatetimeに変換
	temperature_times = [datetime.datetime.strptime(t, "%Y-%m-%d %H:%M") for t in x_temp]
	humidity_times = [datetime.datetime.strptime(t, "%Y-%m-%d %H:%M") for t in x_humidity]

	# グラフの作成
	fig, ax1 = plt.subplots(figsize=(6.4, 4))

	# 温度のプロット（y軸1）
	if len(y_temp) > 0:
		ax1.plot(temperature_times, y_temp, 'r-', label='TEMP[℃]')
		ax1.set_ylabel("TEMP[℃]", color='r')
		ax1.tick_params(axis='y', labelcolor='r')
		# plt.yticks(range(0, 40, 5))	# 表示範囲の指定

	ax2 = None
	if len(y_humi) > 0:
		ax2 = ax1.twinx()
		ax2.plot(humidity_times, y_humi, 'g-', label='HUMID[%]')
		ax2.set_ylabel("HUMID[%]", color='g')
		ax2.tick_params(axis='y', labelcolor='g')

	# 日時フォーマット
	ax1.xaxis.set_major_formatter(mdates.DateFormatter('%H:%M'))
	fig.autofmt_xdate()

	# 凡例の統合
	lines1 = []
	labels1 = []
	lines2 = []
	labels2 = []
	if len(y_temp) > 0:
		lines1, labels1 = ax1.get_legend_handles_labels()
	if len(y_humi) > 0:
		lines2, labels2 = ax2.get_legend_handles_labels()
	ax1.legend(lines1 + lines2, labels1 + labels2, loc='upper left')

	# plt.title("気圧、温度、湿度の変化")
	# plt.grid(True)
	plt.tight_layout()
	if show:
		plt.show()
	if filename:
		fig.savefig(MapsFilesDir+"/"+filename)


# 気圧、温度、湿度のgraphの作成
# X軸：時刻
# Y軸：気圧、温度、湿度の3つを同時に表示。折れ線グラフを使用
def writeEnvGraph1(x_pressure, y_pressure, x_temp, y_temp, x_humidity, y_humi, filename=None, show=False):

	#TODO: 日本語対応するため。どうする？
	# import matplotlib.font_manager as fm
	# font_path = "C:/Windows/Fonts/msgothic.ttc "  # Windows
	# # font_path = "/usr/share/fonts/truetype/ipaexg/ipaexg.ttf"	# Linux
	# jp_font = fm.FontProperties(fname=font_path)
	# plt.rcParams['font.family'] = jp_font.get_name()

	# 文字列をdatetimeに変換
	pressure_times = [datetime.datetime.strptime(t, "%Y-%m-%d %H:%M:%S") for t in x_pressure]
	temperature_times = [datetime.datetime.strptime(t, "%Y-%m-%d %H:%M") for t in x_temp]
	humidity_times = [datetime.datetime.strptime(t, "%Y-%m-%d %H:%M") for t in x_humidity]

	# グラフの作成
	fig, ax1 = plt.subplots(figsize=(6.4, 4))

	# 気圧のプロット（y軸1）
	if len(y_pressure) > 0:
		ax1.plot(pressure_times, y_pressure, 'b-', label='PRES[hPa]')
		ax1.set_ylabel("PRES[hPa]", color='b')
		ax1.tick_params(axis='y', labelcolor='b')

	ax2 = None
	if len(y_temp) > 0 and len(y_humi) > 0:
		ax2 = ax1.twinx()

	# 温度のプロット（y軸2）
	if len(y_temp) > 0:
		ax2.plot(temperature_times, y_temp, 'r-', label='TEMP[℃]')
		ax2.set_ylabel("TEMP[℃] HUMID[%]", color='r')
		ax2.tick_params(axis='y', labelcolor='r')
		plt.yticks(range(0, 100, 5))	# なぜかこれで綺麗になる

	# 湿度のプロット（別のy軸3、上下に重ねる場合は工夫が必要）
	# 簡単に描くためには同じax2に描く
	if len(y_humi) > 0:
		ax2.plot(humidity_times, y_humi, 'g-', label='HUMID[%]')

	# 日時フォーマット
	ax1.xaxis.set_major_formatter(mdates.DateFormatter('%H:%M'))
	fig.autofmt_xdate()

	# 凡例の統合
	lines1 = []
	labels1 = []
	if len(y_pressure) > 0:
		lines1, labels1 = ax1.get_legend_handles_labels()
	lines2 = []
	labels2 = []
	if len(y_temp) > 0 or len(y_humi) > 0:
		lines2, labels2 = ax2.get_legend_handles_labels()
	ax1.legend(lines1 + lines2, labels1 + labels2, loc='upper left')

	# plt.title("気圧、温度、湿度の変化")
	# plt.grid(True)
	plt.tight_layout()
	if show:
		plt.show()
	if filename:
		fig.savefig(MapsFilesDir+"/"+filename)


def writeTestGraph(x, y_pressure, y_temp, y_humi, filename=None, show=False):
	x = toLocalTimeAbs(date2num(x))

	fig = plt.figure(figsize=(8,4), dpi=80)
	axes = fig.add_subplot(111)
	axes.xaxis.set_major_formatter(DateFormatter('%H:%M'))
	axes.xaxis.set_major_locator(mdates.HourLocator(interval=1))
	axes.xaxis.set_minor_locator(mdates.MinuteLocator(interval=10))

	# y軸の設定
	axes.yaxis.set_major_locator(mpl.ticker.MultipleLocator(100))
	axes.yaxis.set_minor_locator(mpl.ticker.MultipleLocator(10))

	plt.fill_between(x, y_pressure, 0, facecolor='blue', interpolate=False, alpha=0.3)
	plt.fill_between(x, y_temp, 0, facecolor='red', interpolate=False, alpha=0.3)
	plt.fill_between(x, y_humi, 0, facecolor='green', interpolate=False, alpha=0.3)

	plt.title("Time - Environment", fontsize=16)
	plt.ylabel('Pressure [hPa], Temperature [C], Humidity [%]')
	plt.xlabel('Time')
	if show:
		plt.show()
	if filename:
		fig.savefig(MapsFilesDir+"/"+filename)

def writeTemplateFile(infile, outfile, vars, encoding='utf8'):
	print("Writing... ", outfile)
	if not os.path.exists(infile):
		print("Not exists:", infile)
		return False

	outf = open(outfile, mode="w", encoding=encoding)
	if not outf:
		print("Open failure: ", outfile)
		return False

	for line in open(infile, encoding='utf_8_sig'):
		while True:
			m = re.match(r'.*\$\((\w+)\)', line)
			if not m:
				break
			var = m.group(1)
			if var in vars:
				val = vars[var]
				val = str(val).replace('\\', '\\\\')	# \d だとre.sub()でgroup indexと解釈されてしまうため
			else:
				print("WARNING: Not found {} in vars".format(var))
				val = ''
			pat = r'\$\(' + var + r'\)'
			if re.match(pat, str(val)):
				print(f"WARNING: may be infinite loop {pat} -> {val}")
				print(f'You should correct this in {infile} or comment.txt')
				input('Press Enter to continue...')
			line = re.sub(pat, str(val), line)
		outf.write(line)
	
	outf.close()

	return True

def buildBasicVars():
	vars = {}
	vars['GeneratorName'] = GENERATOR_NAME
	vars['GeneratorVersion'] = GENERATOR_VERSION
	vars['GeneratorUrl'] = GENERATOR_URL
	vars['GoogleMapKey'] = GOOGLE_MAPS_KEY
	vars['MapWidth'] = MAP_WIDTH
	vars['MapHeight'] = MAP_HEIGHT
	return vars

convert_image = {
    # そのまま
    0: lambda img: img,
    1: lambda img: img,
    # 左右反転
    2: lambda img: img.transpose(Image.FLIP_LEFT_RIGHT),
    # 180度回転
    3: lambda img: img.transpose(Image.ROTATE_180),
    # 上下反転
    4: lambda img: img.transpose(Image.FLIP_TOP_BOTTOM),
    # 左右反転＆反時計回りに90度回転
    5: lambda img: img.transpose(Image.FLIP_LEFT_RIGHT).transpose(Image.ROTATE_90),
    # 反時計回りに270度回転
    6: lambda img: img.transpose(Image.ROTATE_270),
    # 左右反転＆反時計回りに270度回転
    7: lambda img: img.transpose(Image.FLIP_LEFT_RIGHT).transpose(Image.ROTATE_270), 
    # 反時計回りに90度回転
    8: lambda img: img.transpose(Image.ROTATE_90),
}

def makeThumbnails(photos):
	for photo in photos:
		print("Make thumbnail: ", photo['thumb'])
		img = Image.open(photo['file'])
		img.thumbnail((THUMB_WIDTH, THUMB_HEIGHT))
		img = convert_image[photo['ori']](img)
		img.save(photo['thumb'], 'JPEG')

def addAbstractPhotos():
	adds = []
	for i, name in enumerate(RPhotoNames):
		if name in RComment:
			title = RTitle[name] if name in RTitle else ''
			comment = RComment[name]
			if comment.startswith('$'):
				# 新形式のabstract
				abst, comment = parseCommentForAbstract(comment)
				if name == 'abstract':
					# photo以外は特殊処理不要
					RComment[name] = abst
					if comment != '':
						RComment[name] += '\n' + comment
				else:
					# 特殊処理
					# 新形式を旧形式のデータ形式に変換
					newname = name+'a'
					adds.append((i, newname))
					RTitle[newname] = title
					RComment[newname] = abst
					RTitle[name] = '-'
					RComment[name] = comment

	for item in reversed(adds):
		i, name = item
		RPhotoNames.insert(i, name)

# parse abstract from comment
# $で始まる専用
# abst: 概要説明（複数行対応）
# comment: 通常の写真コメント
# Version == 3:
#  $が最初の行の先頭にある場合、abstract開始行とみなす
#    それ以降に改行のみがあればabstractの終了行とみなす（写真コメントもなし扱い）
#    それ以降に$のみの行があればabstractの終了行とみなす（その次の行は写真コメント扱い）
def parseCommentForAbstract(comment):
	abst = ''
	rem = ''	# remaining comment

	multi = comment.startswith('$$')
	lines = comment.split('\n')

	end = False
	for i, line in enumerate(lines):
		if end:
			rem += line+'\n'
		else:
			if multi:
				if abst != "" and line.startswith('$$'):
					# end of multi line
					end = True
					if line != "$$" and line[2:]:
						rem += line[2:] + '\n'
					continue
			else:
				if Version == 3:
					if i == 0:
						# abstract開始行？
						if not line.startswith('$'):	# $で始まるのが前提だからここは来ないのでは？？2025.4.21
							# 写真コメント扱い
							end = True
							rem += line + '\n'
							continue
					else:	# ２行目以降
						if line == '$':
							# abstract終了行
							end = True
							continue
				else:	# Version < 3
					if not line.startswith('$'):
						end = True
						rem += line + '\n'
						continue
			line = re.sub(r'^\$\$?', '', line)
			abst += line + '\n'
	abst = abst.rstrip('\n')
	rem = rem.rstrip('\n')
	return abst, rem

#
# 写真のないコメント用のdummy photoを追加する
# 追加の写真がある場合もここに追加すれば表示される
#
def addDummyPhoto(photos):
	global DummyPhotoBaseId, PhotoIds

	names = []

	if Version == 0:
		if 'photo' in RTitle:
			names.append(RTitle['photo'])

		if 'photo' in RComment:
			names += RComment['photo'].split("\n")

	else:	# Version >= 2
		# RPhotoNamesにあってphotos[]にないものを追加
		for name in RPhotoNames:
			if name.startswith(';') or name in PredefinedSection:
				continue
			found = False
			for photo in photos:
				if name == photo['name']:
					found = True
					break
			if not found:
				names.append(name)

	for name in names:
		photo_id = str(DummyPhotoBaseId)
		DummyPhotoBaseId += 1
		PhotoIds.append(photo_id)
		PhotoBaseNames[photo_id] = name

	return photos

def buildVarsForHtml(points, photos, stat):
	global RPhotoNames, RTitle, RComment
	global PhotoIds, PhotoBaseNames, BottomComment, PhotoIdToIndex
	global CurYear, CurMon, CurDay
	global Properties
	global EnvGraphHidden
	global CostHeaderHidden

	# 変数の作成

	if 'date' in RTitle:
		date = RTitle['date']
	else:
		date = CurDateStr
	m = re.match(r'(\d+)/(\d+)/(\d+)', date)
	if not m:
		print(f"Invalid date format: {date} should be YYYY/MM/DD")
		exit(1)
	year = int(m.group(1))
	month = int(m.group(2))
	day = int(m.group(3))
	filedate = "{:4d}{:02d}{:02d}".format(year, month, day)

	_date = datetime.datetime(year=year, month=month, day=day)
	weekday = _date.weekday()
	holiday = jpholiday.is_holiday(_date)
	weekdaystr = ['月', '火', '水', '木', '金', '土', '日'][weekday]
	color = 'FF8080' if holiday else 'FF0000' if weekday==6 else '0000FF' if weekday == 5 else ''
	if color:
		weekdayhtml = '<font color="#'+color+'">' + weekdaystr + '</font>'
	else:
		weekdayhtml = weekdaystr

	title = filedate
	if 'title' in RTitle:
		title = RTitle['title']

	datelinks = ''
	if 'prev_date' in RTitle:
		comment = RComment['prev_date'] if 'prev_date' in RComment else ''
		datelinks += " " + makeLink(RTitle['prev_date'], comment, 'prev_date', CurDate)
	if 'next_date' in RTitle:
		comment = RComment['next_date'] if 'next_date' in RComment else ''
		datelinks += " " + makeLink(RTitle['next_date'], comment, 'next_date', CurDate)

	otherlinks = ''
	yamap_url = ''
	if 'yamap' in RTitle:
		yamap_url = RTitle['yamap']
		otherlinks += " " + makeLink(' [YAMAP活動日記]', yamap_url) 

	rest_timedt = None
	rest_timedts = ''
	actual_movingdt = None
	movingdt = None
	actual_movingdts = ''
	if points:
		startdts = toLocalTime(points[0]['time']).strftime('%Y/%m/%d %H:%M')
		enddts = toLocalTime(points[-1]['time']).strftime('%Y/%m/%d %H:%M')
		movingdt = points[-1]['time'] - points[0]['time']
		movingdts = toHHMM(movingdt)
		if 'rest_time' in stat:
			rest_timedt = datetime.timedelta(seconds=stat['rest_time'])
			rest_timedts = toHHMM(rest_timedt)
			actual_movingdt = movingdt - rest_timedt
			actual_movingdts = toHHMM(actual_movingdt)

	for i, photo in enumerate(photos):
		id = "photoID"+str(i+1)
		PhotoIds.append(id)
		base = photo['name']
		PhotoBaseNames[id] = base
		PhotoIdToIndex[id] = i

	# カロリー計算
	# https://www.yamakei-online.com/iframe/help.php?kind=course_const
	course_const = 0
	kcal = 0
	if actual_movingdt and stat['dist']:
		course_const = 1.8 * actual_movingdt.seconds/3600 + 0.3 * stat['dist'] + 10 * stat['total_up']/1000 + 0.6 * stat['total_down']/1000
		weight = BODY_WEIGHT + LOAD_WEIGHT
		kcal = course_const * weight
	# print(f"コース定数={course_const}")
	# print(f"消費カロリー={kcal}")

	out = ''
	out_cost = ''

	vars = buildBasicVars()

	vars['AbstractTitle'] = Properties['AbstractTitle']
	vars['Abstract'] = printAbstract()
	if 'abstract' in RTitle:
		vars['AbstractTitle'] = RTitle['abstract']

	sortPhotos()
	out += printPhotos(photos)

#	printArray(\@TGraph);
	# Additional sections
	if 'add' in RTitle:
		out += printSection('add')
	
	if 'cost' in RTitle:
		s, total = printTable('cost')
		out_cost += s
		if s == '':
			CostHeaderHidden = 'hidden'
		vars['TotalCost'] = comma(total)
	
	if 'comment' in RTitle:
		out += printSection('comment')

	# error対策
	if not 'car' in stat:
		stat['car'] = False

	vars['car'] = stat['car']
	vars['Title'] = title
	vars['FileDate'] = filedate
	vars['Date'] = date
	vars['Weekday'] = weekdaystr
	vars['WeekdayHtml'] = weekdayhtml
	vars['DateLinks'] = datelinks
	vars['OtherLinks'] = otherlinks
	vars['YamapUrl'] = yamap_url

	if points:
		vars['StartDateTime'] = startdts
		vars['EndDateTime'] = enddts
		vars['MovingTime'] = movingdts
		vars['ActualMovingTime'] = actual_movingdts
		vars['RestTime'] = rest_timedts
		vars['MoveDistance'] = round(stat['dist'], '0.00')	# [km]
		vars['AvgSpeed'] = round(stat['avgspeed'], '0.00')	# [km/h]
		vars['ActualAvgSpeed'] = round(stat['actualavgspeed'], '0.00')	# [km/h]
		vars['RelAlt'] = comma(int(stat['maxalt'] - stat['minalt']))	# [m]
		vars['MaxAlt'] = comma(int(stat['maxalt']))	# [m]
		vars['MinAlt'] = comma(int(stat['minalt']))	# [m]
		vars['NumOfTrack'] = len(points)
		vars['NumOfPoints'] = len(points)
		vars['TotalAlt'] = comma(int(stat['total_up'] + stat['total_down']))
		vars['TotalUp'] = comma(int(stat['total_up']))		# m
		vars['TotalDown'] = comma(int(stat['total_down']))	# m
		vars['TotalUpTime'] = comma(int(stat['total_up_time'] / 60))	# min
		vars['TotalDownTime'] = comma(int(stat['total_down_time'] / 60))	# min
		vars['TotalUpDistance'] = round(stat['total_up_dst'], '0.00')
		vars['TotalDownDistance'] = round(stat['total_down_dst'], '0.00')
		vars['TotalUpAvg'] = round(stat['total_up_avg'], '0.0')		# km/h
		vars['TotalDownAvg'] = round(stat['total_down_avg'], '0.0')	# km/h
		vars['NumOfPeaks'] = comma(stat['peak_count'])
		vars['MinElev'] = MinElev
		vars['MinElevPeak'] = MinElevPeak
		vars['MinElevUpDown'] = MinElevUpDown

	vars['CourseTime'] = '-'
	vars['kcal'] = str(int(kcal)) if kcal else '-'
	vars['TimeRatio'] = '-'
	vars['TimeRatioPerc'] = '-'
	vars['ActualTimeRatio'] = '-'
	vars['ActualTimeRatioPerc'] = '-'
	vars['CourseConst'] = '-'
	vars['CourseConstLevel'] = '-'

	if 'coursetime' in RTitle:
		ctime = RTitle['coursetime']
		if ctime.startswith('cc'):	# コース定数によるコースタイム算出
			if movingdt:
				coursetime = calcCourseTimeFromCourseConst(stat, int(ctime[2:]))
				hours = coursetime.seconds // 3600
				minutes = (coursetime.seconds // 60) % 60
				vars['CourseTime'] = f'{hours}:{minutes:02}'
		else:
			ctime = datetime.datetime.strptime(RTitle['coursetime'], '%H:%M')
			coursetime = datetime.timedelta(hours=ctime.hour, minutes=ctime.minute)
			vars['CourseTime'] = RTitle['coursetime']
		if movingdt:
			if coursetime.seconds > 0:
				vars['TimeRatio'] = round(movingdt / coursetime, '0.00')
				vars['TimeRatioPerc'] = int(coursetime / movingdt * 100)
				if actual_movingdt:
					vars['ActualTimeRatio'] = round(actual_movingdt / coursetime, '0.00')
					vars['ActualTimeRatioPerc'] = int(coursetime / actual_movingdt * 100)
				if ctime is str and ctime.startswith('cc'):	# コース定数の指定
					cc = int(ctime[2:])
					vars['CourseConst'] = cc
					vars['CourseConstLevel'] = calcCourseConstLevel(cc)
				else:
					vars['CourseConst'], vars['CourseConstLevel'] = calcCourseConst(stat, coursetime)

	if 'kcal' in RTitle:
		vars['kcal'] = comma(int(RTitle['kcal']))

	# Peak & 休憩一覧
	if 'peaks' in stat and 'rests' in stat:
		s = ''
		sum_rest_time = 0
		pandr = stat['peaks'] + stat['rests'] + stat['mounts']
		pandr.sort(key=lambda x:x['index'])
		peak_index = 0
		prev_alt = None
		for item in pandr:
			pt = points[item['index']]
			dt = toLocalTime(pt['time'])
			s += '{}:{:02d}'.format(dt.hour, dt.minute) + ' '
			if 'alt' in item:
				alt = item['alt']
				if 'name' in item:
					# mounts情報
					s += '<b>{}</b> ({}m)'.format(escapeHtml(item['name']), int(alt))
				else:
					mark = ''
					if not prev_alt:
						prev_alt = points[0]['alt']
					mark = '↑' if alt > prev_alt else '↓'
					prev_alt = alt
					s += ' {}m {}'.format(int(alt), mark)
					peak_index += 1
			if 'dur' in item:
				dur = item['dur']
				sum_rest_time += dur
				s += '休憩 ({}分<small>{:02d}</small>)'.format(dur//60, dur%60)
			s += '\n'
		s += '合計休憩時間：' + '{}分<small>{:02d}</small>'.format(sum_rest_time//60, sum_rest_time%60)
		vars['Peaks'] = s
		vars['RestTimeMin'] = sum_rest_time//60
	else:
		vars['Peaks'] = ''

	vars['EnvGraphHidden'] = EnvGraphHidden
	vars['EnvGraphAllHidden'] = EnvGraphAllHidden
	vars['EnvGraphPressHidden'] = EnvGraphPressHidden
	vars['EnvGraphTempHidden'] = EnvGraphTempHidden
	vars['CostHeaderHidden'] = CostHeaderHidden
	vars['PhotoArea'] = out.replace('\\', '￥')
	vars['CostArea'] = out_cost.replace('\\', '￥')

	savePhotos(vars, photos)

	return vars

# コース定数の計算
def calcCourseConst(stat, coursetime):
	cc = int(1.8 * coursetime.seconds/3600 + 0.3 * stat['dist'] + 10.0 * stat['total_up'] / 1000 + 0.6 * stat['total_down'] / 1000)
	return cc, calcCourseConstLevel(cc)

def calcCourseConstLevel(cc):
	if cc < 12:
		return 'やさしい'
	elif cc < 25:
		return 'ふつう'
	else:
		return 'きつい'

# コース定数からコースタイムの計算
# timedeltaで返す
def calcCourseTimeFromCourseConst(stat, courseconst):
	# ct = (cc - (0.3 x d + 10 x up + 0.6 x dn)) / 1.8 [h]
	ct = (courseconst - (0.3 * stat['dist'] + 10.0 * stat['total_up'] / 1000 + 0.6 * stat['total_down'] / 1000)) / 1.8
	return datetime.timedelta(hours=ct)

def makeLink(label, url, _type='', date=None):
	if label == 'auto':
		if not date:
			print("Warning : makeLink() : no date provided.")
			return ''
		if _type == 'prev_date':
			date -= datetime.timedelta(days=1)
			label = "[前日 {}/{}]".format(date.month, date.day)
			url = "../{:04d}{:02d}{:02d}/".format(date.year, date.month, date.day)
		else:
			date += datetime.timedelta(days=1)
			label = "[翌日 {}/{}]".format(date.month, date.day)
			url = "../{:04d}{:02d}{:02d}/".format(date.year, date.month, date.day)
	return "<a href=\"{}\">{}</a>".format(url, label)

# photos[x]の'time'順に並び替える
# 'time'がない場合は、photos内の直前の時刻に合わせる
# 時刻が一つも見つからない場合は何もしない
def sortPhotosByTime(photos):
	lasttime = None
	# まずは先頭の時刻を探す
	for photo in photos:
		if 'time' in photo:
			lasttime = photo['time']
			break

	if not lasttime:
		return	# 時刻が一つも見つからない

	# 'time'が無いitemは前itemと同じ時刻をassign
	for photo in photos:
		if 'time' in photo:
			lasttime = photo['time']
		else:
			photo['time'] = lasttime

	# sort
	photos.sort(key=lambda x: x['time'])

# PhotoIdsの順番を@RPhotoNames順に並び替える
# RPhotoNamesに存在しないものはそれより一番近い直前のものと連結する
def sortPhotos():
	global PhotoIds, PhotoBaseNames

	temp = []

	# PhotoBaseNames -> PhotoIds index mappingを作る
	NameToIndex = {}	# name -> index on PhotoIds
	for index, id in enumerate(PhotoIds):
#		print("{}:photoid={}".format(index, id))
		name = PhotoBaseNames[id]
		if not name in NameToIndex:
			NameToIndex[name] = index
			# 最初に出てきた方を優先する
			# photo sectionに追加したほうを優先するため
			# そうしないと追加されたphotoではなく、通常のphotoとして扱われてしまうため

	# RPhotoNamesに存在するPhotoIds indexを調べる
	ExistIndex = {}
	for name in RPhotoNames:
		if name in NameToIndex:
			index = NameToIndex[name]
			ExistIndex[index] = True

	bottom_index = -1
	for name in RPhotoNames:
		if name in NameToIndex:
			index = NameToIndex[name]
			if False:
				# これ、何のため必要？ -> RPhotoNamesに存在しない場合の対応？
				# RPhotoNamesに無いPhotoIds上のtop indexを探す
				top = -1
				i = index - 1
				while i>=0:
					if i in ExistIndex:
						break
					top = i
					i -= 1

				# 見つかったidを追加
				if top>=0:
					print("found top", top)
					i = top
					while i<index:
						print("append1", i, PhotoIds[i])
						temp.append(PhotoIds[i])
						i += 1

			temp.append(PhotoIds[index])
			if bottom_index < index:
				bottom_index = index

	# 残りのPhotoIdsを追加する
	for i in range(bottom_index+1, len(PhotoIds)):
		temp.append(PhotoIds[i])

	PhotoIds = temp

def printAbstract():
	s = ''

	key = 'abstract'

	comment = RComment[key] if key in RComment else ''
	comment = re.sub(r'^\n', '', comment)	# 先頭の改行のみを削除

	comment, n = re.subn(r'^\/', '', comment)
	if comment == '':
		return ''

	comment = replaceNewLine(comment)
	s += '''
	<td colspan="1" valign="middle">
		{}
	</td>
	</tr>
	'''.format(comment)

	return s

def printPhotos(photos):
	s = ''

# 	print << "EOH"
# <div id="photos">
# <table width="$Width">
# 	<tr><td bgcolor="#404040" align="left">
# 		<font color="#F0F0F0"><b>写真</b></font>
# 	</td></tr>

# </table>
# <small>写真Click→拡大 住所Click→Google Mapsへjump 写真拡大後：画像Click→閉じる</small>
# <table width="$Width" border="0" bordercolor="#FFFFFF">
# EOH

	col = 0
	col_ids = []	# id list
	for id in PhotoIds:
		# print("id: ", id, PhotoBaseNames[id])
		if existPhotoHeader(id):
			if len(col_ids) == 1:
				# １つだけの場合は横コメント
				s += printPhoto(col_ids[0], photos)
			else:
				s += printPhotoArray(col_ids, photos)

			col_ids = []
			col = 0
			s += printPhotoHeader(id)

		title, comment = getPhotoComment(id)

		if title.startswith('*'):	# 非表示
			s += printHiddenPhoto(id)
			continue

		# コメント内容のファイル出力
		base = PhotoBaseNames[id]
		if base in MarkedFiles:
			c = comment.lstrip('/')
			if title == '' or title == '-':
				MarkedFileComments[base] = c + '\n'
			else:
				MarkedFileComments[base] = title + '\n' + c + '\n'

		no_id = False
		col += 1
		comment, n = re.subn(r'^\/', '', comment)
		no_btm_comment = (n > 0)
		if comment == '' or (not no_btm_comment and len(comment) < MaxCommentChars):
			if title == '' or title == '-' or len(col_ids)==0:
				if comment != '':
					BottomComment[id] = True

				# title無し、titleがあっても左端のphotoであればOK
				col_ids.append(id)
				if col < PhotoColumns:
					continue
				no_id = True

		# print buffered photos
		s += printPhotoArray(col_ids, photos)
		col_ids = []
		col = 0
		if not no_id:
			s += printPhoto(id, photos)

	if col!=0:
		s += printPhotoArray(col_ids, photos)

	id = 'last'
	if id in RComment:
		PhotoBaseNames[id] = id
		s += printCommentTable(id)

	return s

def printPhotoHeader(photo_id):
	s = ''

	photo_basename = PhotoBaseNames[photo_id]
	key = photo_basename
	title = RTitle[key] if key in RTitle else ''

	# Abstract
	abstract = False
	if key+'a' in RComment or key+'a' in RTitle:
		s += printComment(None, key+'a', 1)
		abstract = True

	if title.startswith('*'):	# 非表示
		return ''

	elif title == '':
#		title = photo_basename
		pass

	if title != '' and title != '-':
		if abstract:
			s += '''
	<tr>'''
		else:
			s += '''
	<tr id="{}">'''.format(photo_id)

		s += '''
		<td colspan="{}" bgcolor="#C5FFE0">
			<b>{}</b>
		</td>
	</tr>'''.format(PhotoColumns, title)

	return s

# 列並びのphotoをbreakするかどうかで判断する
def existPhotoHeader(photo_id):
	photo_basename = PhotoBaseNames[photo_id]
	key = photo_basename

	if key+'a' in RComment or key+'a' in RTitle:
		return True

	title = RTitle[key] if key in RTitle else ''

	if title.startswith('*'):
		return False

	if title != '' and title != '-':
		return True

	if key in RComment:
		if RComment[key].startswith('/'):
			RComment[key] = re.sub(r'^\/\/', '', RComment[key])
			return True

	return False

def printPhotoArray(col_ids, photos):
	# print("printPhotoArray: ", len(col_ids))
	global PhotoIdToIndex

	if len(col_ids) <= 0:
		return ''

	s = "\n<tr>"
	for id2 in col_ids:
		s += '''
  <td valign="top">
  <table>'''
		s += printPhoto(id2, photos, id2 in BottomComment)
		s += '''
  </table>
  </td>'''

	s += "\n</tr>"

	return s

def getPhotoById(photo_id, photos):
	if photo_id in PhotoIdToIndex:
		return photos[PhotoIdToIndex[photo_id]]
	return None

# Input:
#	photo_id
#	PhotoBaseNames
#	PhotoDate
#	PhotoPos - optional
#	PhotoLoc - optional
#	RTitle
#	RComment
#
#
def printPhoto(photo_id, photos, btm_comment=False):
	# print("printPhoto: id={} btm={}".format(photo_id, btm_comment))
	s = ''

	photo = getPhotoById(photo_id, photos)
	if not photo:
		photo = {}

	photo_no = re.sub(r'[^\d]', '', photo_id)	# 数字以外を削除
	
	photo_basename = ''
	if photo_id == '':
		print("no photo_id!!")
	else:
		if photo_id in PhotoBaseNames:
			photo_basename = PhotoBaseNames[photo_id]
		else:
			print("no photo_basename!! : ", photo_id)

	key = photo_basename

	comment = RComment[key] if key in RComment else ''

	comment = replaceNewLine(comment)
	comment, n = re.subn(r'^\/', '', comment)
	if comment == '' or n > 0:
		btm_comment = False

	photo_link = ''
	if 'file' in photo:
		photo_link = photo['file']

	photo_alt = key

	if 'lng' in photo and photo['lng']:
		lngs = lngToDMS(photo['lng'])
		lats = latToDMS(photo['lat'])
		photo_alt += "\n{}, {}".format(lngs, lats)
		if 'alt' in photo:
			photo_alt += ', {}m'.format(round(photo['alt'], '0.0'))

	photo_loc = ''
	if 'addr' in photo and photo['addr']:
		photo_loc = photo['addr']

	if photo_loc != '':
		photo_alt += "\n" + photo_loc
	else:
		if 'lng' in photo and photo['lng']:
			photo_loc = '(no address)'

	photo_time = ''
	if 'time' in photo and photo['time']:
		dt = toLocalTime(photo['time'])
		photo_time = "{}:{:02d}".format(dt.hour, dt.minute)

	# 累積移動距離
	# km単位で小数第一位で表示する
	photo_distance = ''
	if 'cum' in photo:
		if photo['cum'] >= 0.1:
			photo_distance = ' {:.1f}km'.format(photo['cum'])
		else:
			photo_distance = ' 0km'

	photo_altitude = ''
	if 'alt' in photo:
		photo_altitude = '{:,}m'.format(int(photo['alt']))

	photo_dist_alt = ''
	if ShowPhotoDistAlt and (photo_distance or photo_altitude):
		photo_dist_alt = ' ' + photo_distance + '/' + photo_altitude

	photo_filename = ''
	if PrintPhotoName and photo:
		photo_filename = " " + photo['name']

	if existPhotoHeader(photo_id):
		s += '''
	<tr>'''
	else:
		s += '''
	<tr id="{}">'''.format(photo_id)

	jump_photo_marker = "javascript:jumpphotomarker({});".format(photo_no)

	colspan = PhotoColumns - 1

	s += '''
		<td>'''

	map_pop_id = "mapPop" + photo_no

	use_shadow = False	# experimental (イマイチ)

	if photo_link:
		photo_mouseover = ''
		photo_mouseout = ''
		if 'lng' in photo and photo['lng']:
			photo_mouseover = "showMapPop('{}', {}, {})".format(map_pop_id, photo['lat'], photo['lng'])
			photo_mouseout = "hideMapPop('{}')".format(map_pop_id)
		div_start = ''
		div_end = ''
		if use_shadow:
			div_start = '<div class="shadow">'
			div_end = '</div>'
		s += f'''
			<a href="{photo_link}" rel="lightbox[photo]">{div_start}<img src="{photo['thumb']}" alt="{photo_alt}">{div_end}</a><br>
			<div onMouseover="{photo_mouseover}" onMouseout="{photo_mouseout}">
			<font size="-2"><a href="{jump_photo_marker}">{photo_loc}</a></font>
			</div>'''

		if PrintPhotoName or not btm_comment:
			# 時：分 ファイル名
			ss = photo_time + photo_filename + photo_dist_alt;
		else:
			ss = photo_time	+ photo_dist_alt # 時刻は常に表示

		s += '''
			<center><small>{}</small></center>'''.format(ss)
	else:
		# dummy fileであっても存在する場合は表示する
		file = photo_basename + ".jpg"
		if os.path.exists(file):
			s += '''
			<a href="{}" rel="lightbox[photo]"><img src={} width="320" alt="{}"></a>'''.format(file, file, file)

	if btm_comment:
		s += '''
			<p><center>
			{}
			</center></p>'''.format(comment)

	if photo_link:
		s += '''
			<div id="{mapPopId}" style="-moz-border-radius: 7%; background:gray; position:absolute; display:none;margin-top: 5px;margin-right: 5px;margin-left: 5px;margin-bottom: 5px; width:250px; height:250px;">
			</div>'''.format(mapPopId=map_pop_id)

	s += '''
		</td>'''

	if not btm_comment:
		s += '''
		<td colspan="{}" valign="middle">
			{}
		</td>'''.format(colspan, comment)

	s += '''
	</tr>'''

	if key+'s' in RComment:
		s += printComment(None, key+'s', None)

	return s

# Print comment with table.
def printCommentTable(photo_id):
	s = ''

	if photo_id == '':
		print("Warning: no photo_id!!")

	s += '''
<div id="{}">
<table width="{}" border="0" bordercolor="#FFFFFF">
'''.format(photo_id, Width)

	s += printComment(photo_id, None, None)

	s += '''
</table>
</div>'''

	return s

# Input:
#	photo_id or photo_basename
#	PhotoBaseNames
#	RTitle
#	RComment
#
def printComment(photo_id, photo_basename, colspan):
	s = ''

	if photo_id:
		if photo_id in PhotoBaseNames:
			photo_basename = PhotoBaseNames[photo_id]
		else:
			print("no photo_basename!!")

	key = photo_basename
	title = RTitle[key]
	comment = RComment[key] if key in RComment else ''
	comment = re.sub(r'^\n', '', comment)	# 先頭の改行のみを削除
	photo_alt = key

	if title.startswith('*'):	# 非表示
		return ''

	elif title == '':
#		title = photo_basename
		pass

	comment = replaceNewLine(comment)
	comment = comment.replace('\t', '　')	# タブは全角空白に置き換え

	s += '''
	<tr>'''

	if title != '' and title != '-':
		s += '''
		<td colspan="{}" bgcolor="#C5FFE0">
			<b>{}</b>
		</td>
	</tr>
	<tr>'''.format(PhotoColumns, title)

	if colspan:
		s += '''
		<td colspan="{}" valign="middle">
			{}
		</td>'''.format(PhotoColumns, comment)
	else:
		cols = PhotoColumns-1
		s += '''
		<td>
		</td>
		<td colspan="{}" valign="middle">
			{}
		</td>'''.format(cols, comment)

	s += '''
	</tr>'''

	return s

def savePhotos(vars, photos):
	s = ''

	rtitle = {}
	rcomment = {}
	markedfiles = []
	if os.path.exists(PhotoCommentFile):
		_, _, rtitle, rcomment, markedfiles = t2g3lib.parseCommentFile(PhotoCommentFile, no_br=True)

	def getUserComment(id):
		photo = getPhotoById(id, photos)
		if not photo: return '', False

		if 'cmt' in photo:
			cmt = photo['cmt']
			if cmt and len(cmt) > 0:
				if cmt.startswith('>'):
					cmt = cmt[1:]
					return cmt, True
				else:
					return cmt, False
		return '', False

	def parseComment(basename, rtitle, rcomment, markedfiles, cmt, marked):
		if basename in rtitle:
			if rtitle[basename] == '*':
				cmt = '*\n'
			else:
				cmt = rtitle[basename] + '\n'
				if basename in rcomment and rcomment[basename]:
					cmt += rcomment[basename] + '\n'
			if basename in markedfiles:
				marked = True
		return cmt, marked

	for id in PhotoIds:
		basename = PhotoBaseNames[id]

		# exifのUserCommentがある場合は追加する
		cmt, marked = getUserComment(id)
		cmt, marked = parseComment(basename, rtitle, rcomment, markedfiles, cmt, marked)

		s += basename + ('>' if marked else '') + '\n'

		if cmt: s += cmt
		s += '\n'

	vars['PhotoBaseNames'] = s

	# for markdown
	s = ''
	for id in PhotoIds:
		basename = PhotoBaseNames[id]

		# exifのUserCommentがある場合は追加する
		cmt, marked = getUserComment(id)
		cmt, marked = parseComment(basename, rtitle, rcomment, markedfiles, cmt, marked)

		s += '![](' + PhotoBaseNames[id]+'.jpg' + ('*' if cmt=='*' else '') + ')'+'\n'
		s += PhotoBaseNames[id] + ('>' if marked else '') + '\n'

		if cmt: s += cmt
		s += '\n'

	vars['MDPhotoNames'] = s

	vars['IncludePhotos'] = 'include '+PhotosMD if MarkdownMode else vars['PhotoBaseNames']

	comtplfile = CommentTemplFileCar if vars['car'] else CommentTemplFileWalk

	writeTemplateFile(SCRIPT_PATH + '/' + comtplfile, PhotoNamesFile, vars)

	if MarkdownMode and not os.path.exists(PhotosMD):
		writeTemplateFile(SCRIPT_PATH + '/' + PhotosMDTemplFile, PhotosMD, vars)

	if not os.path.exists(CommentFile):
		shutil.copy2(PhotoNamesFile, CommentFile)

def getPhotoComment(photo_id, suffix=''):
	photo_basename = PhotoBaseNames[photo_id]
	key = photo_basename + suffix
	title = RTitle[key] if key in RTitle else ''
	comment = RComment[key] if key in RComment else ''
	return title, comment

def printHiddenPhoto(id):
	# Google mapからのjump用のみ
	return '''
	<div id="{}">
	</div>'''.format(id)

def printSection(name):
	title = RTitle[name]
	comment = RComment[name] if name in RComment else ''

	return printSection2(name, title, comment)

def printSection2(name, title, comment, bgcolor=None):

	if not bgcolor:
		bgcolor = "#404040"

	s = '''
<!-- Section {name} -->
<div id="{name}">
<table width="{Width}">
	<tr><td bgcolor="{bgcolor}" align="left">
		<font color="#F0F0F0"><b>{title}</b></font>
	</td></tr>
</table>'''.format(name=name, Width=Width, bgcolor=bgcolor, title=title)

	if comment:
		s += '''
<table width="{Width}" bgcolor="#CCCCCC" bordercolor="#000000" border="1">
<tr>
<td>
{comment}
</td>
</tr>
</table>'''.format(Width=Width, comment=comment)

	s += '''
</div>'''

	return s

def printTable(name):
	title = RTitle[name]
#	comment = RComment[name];

	s = '''
<div id="{name}">
<table width="{Width}">
	<tr><td bgcolor="#404040" align="left">
		<font color="#F0F0F0"><b>{title}</b></font>
	</td></tr>
</table>

<table width="{Width}" border="1">
'''.format(name=name, Width=Width, title=title)

	lines = []
	if name in RComment:
		lines = RComment[name].split("\n")

	# 合計金額を求める
	total = 0
	if name == 'cost':
		total, rows = t2g3lib.parseCost(lines, True)
		if total>0:
			lines.append("合計\t\t{}".format(total))
		# 正味のデータが無ければ空にする
		if rows == 0:
			return '', 0

	for line in lines:
		items = line.split("\t")
		s += '''
  <tr>'''
		for item in items:
			if isascnum(item):
				item = comma(item)
			s += "\n    <td>{}</td>".format(item)

		s += '''
  </tr>'''

	s += '''
</table>
</div>'''
	return s, total

# HTML用に改行<br>を追加する
# ただし、<table>などがあると期待した結果にならないため、
# 最後がend tagの場合は<br>を追加しない
# 先頭にtabがある場合は<blockquote>で囲み、先頭にtabがある行が連続する場合はそれらをまとめて<blockquote>で囲む
def replaceNewLine(text):
	# 行単位に分解する
	lines = text.split('\n')
	out = ''
	multi = False	# 複数行対応
	# <html>～</html>は<br>を付加しない
	html = False
	blockquote = False

	# 各行を処理する
	for line in lines:
		if multi:
			if line == '$$':
				multi = False
				continue
		else:
			multi = line.startswith('$$')
			if multi:
				line = line[2:]
				if line == '':
					continue	# $$のみは無視する

		if line == '<html>':
			html = True
			continue
		elif line == '</html>':
			html = False
			continue

		if line.startswith('\t'):
			if not blockquote:
				out += '<blockquote>'
				blockquote = True
			line = line[1:]
		elif blockquote:
			out += '</blockquote>'
			blockquote = False

		s = line.replace(' ', '')
		s = s.replace('\t', '')
		# 行末が>の場合はend tagとみなす
		# それ以外は<br>を追加する
		if html or s.endswith('>'):
			out += line + '\n'
		else:
			out += line + '<br>\n'
	return out

# CSVファイルを読み込む
# １列目：date timeデータ
# column列目：必要な数値データ
# 戻り値：
# 	date timeデータ, 数値データ
def read_csv(filename, column, header=False):
    dates = []
    values = []
    with open(filename, 'r', encoding='utf-8') as f:
        for line in f:
            if header:
                header = False
                continue
            line = line.rstrip('\n')
            items = line.split(',')
            if len(items) > column:
                dates.append(items[0])
                values.append(items[column])
    return dates, values

def isascnum(s):
    return True if s.isdecimal() and s.isascii() else False

def comma(val):
	return '{:,}'.format(int(val))

def escapeHtml(s):
	s = s.replace('&', '&amp;')
	s = s.replace('<', '&lt;')
	s = s.replace('>', '&gt;')
	s = s.replace('"', '&quot;')
	return s

#
# Helpers
#

def str2datetime(str):
	try:
		return datetime.datetime.strptime(str, "%Y-%m-%dT%H:%M:%S%z")
	except:
		return datetime.datetime.strptime(str, "%Y-%m-%dT%H:%M:%S.%f%z")

def round(val, fmt):
	return float(Decimal(val).quantize(Decimal(fmt)))

def pointsToLocs(points):
	locs = []
	for pt in points:
		locs.append((pt['lat'], pt['lng']))
	return locs

def toLocalTimeAbs(abstime):
	return abstime + (LOCALTIME_DIFF * 3600)/86400

def toLocalTime(dt):
	return dt + datetime.timedelta(hours=LOCALTIME_DIFF)

def toHHMM(delta):
	h, m = divmod(delta.seconds, 3600)
	m = m // 60
	return '{}:{:02d}'.format(h, m)

# 指定単位[sec]にdtを丸める(最大24Hまで)
def roundTime(dt, unit):
	sec = dt.hour * 3600 + dt.minute * 60 + dt.second
	asec = sec // unit * unit
	hour = asec // 3600
	minute = asec // 60 % 60
	second = asec % 60
	dt = dt.replace(hour=hour, minute=minute, second=second, microsecond=0)
	return dt


def test(files):
	# readProperties()
	points = []
	for file in files:
		base, ext = os.path.splitext(file)
		print(file)
		files = glob.glob(file)
		if ext == '.gpx':
			for file in files:
				# parse_gpx(file)
				points += read_gpx(file)
				# locs = pointsToLocs(points)
				rests = None
				dist_cum, alt_min, alt_max = calcDistanceVelocity(points)
				peaks, total_up, total_dn, peak_count, total_up_dst, total_dn_dst, total_up_time, total_dn_time = getPeaks(points, rests, MinElevUpDown)
				# for p in points:
				# 	print(p['alt'])
				print("total_up={} total_dn={}".format(total_up, total_dn))

	exit()

	# encoded 
	# r = polyline.decode('{ekwEs|dqYOBOJ]Ta@X]^[^CXAr@CLITGf@G^Gd@Cf@CVIf@Ml@M^QZ?D?v@Dr@@j@Dj@Al@Ah@?DCbA?VBJ@n@MdAL`@J^TLLBNVFh@F`@?NBX@j@ALMv@Yh@_@`@e@n@M`@?D@d@BPJh@Rf@Vf@Lr@Fx@?`A@f@Nh@T^T^L^@h@?r@@f@Cb@?v@Pb@N\\Td@Pb@JZFh@@`@@XBp@@h@BZBb@B`@BVADCGILCXMd@Ml@QXOJe@d@K^M\\E`@E`@Ud@SXYVGFGNI\\SVGBU?QW]W_@VQLKRQRMNGV?n@?d@Gb@KZM\\OVIXGb@Cb@?BGz@Fr@FZFFH^Jh@`@b@RR@@f@VCp@IPEh@Jf@@\\BZ_@b@Qf@IDe@ZCb@NNRNDDJXDXDXDf@DZ?VC\\A^Hb@LJZ?ZEV`@?P@THRLDDP?ZFVBTDTBVJNHLDNCVCZELSD[WQW]W_@?Lx@Bt@AVK^Gb@C`@Eb@Tr@F\\?p@A`@GVI`@s@RYZEHX?')
	# r = polyline.decode('sv|vEmutoYUJBR]X?GWTDKSAF?DOC?T?IQCQ@?OPSFIDCDKHEBDGFSBOIGIFOBKPEJKJOECQ?OFQDKBIBS?KI?GFGDSHOLGLCPCFCQAAFUAOFS?UIEIJ??IDINIPMRGXEJECAMEYIMIQKEALGTEFIZKJETQDEE??GIEK@AIQODEDA?M@GA@K?KCMDS?K?IAKKEA@QLILOHC@OJGDAB????GHKFBHBF?HEFSNSPO\\OTQDMEMMQLOJOPWLc@@e@Bg@DSDg@L]JWHSHQDUKSCQ?MCEBCBK@EAIIK@@LIDIEGEKFKAKCKIIQY]CBEVM?QEGE?_@E]QKWBW?S@e@?YBMEKM?BEKIWOYOc@Mc@Qk@W_@GQIUS]O_@WY[_@Qc@CB@II[Kg@KMOW@C@?C?IBc@D??A??C@?@AB?A???B?AC?BAC??????@D?A?C?A@?C@@AA@????????@?ABC?AD?CA?JOJQ^KFGPUBKA?B@?EETINSUOOOGIKc@D[LGDUNa@EUKA?SQk@e@W_@Y_@UAKIQGUUACCHAABGAFHDZJHDF@NBLDX`@RRX^XVNJNFRBPINGXK`@KFEKq@Gi@Ee@Ge@C_@Ce@Mk@A]Ca@BKDOJYDg@G[Sc@KQAIIU@SIQMk@Mc@G]AIJ[GEA?CG@OCWAa@?m@Iu@Km@??@g@HWJ[FU@BGVIVIZ?B?PDf@Fd@D`@?f@?b@K`@EGIYKs@Os@Ie@IMIBCB?RVl@BHBh@D`@F`@HRCDINGDUf@Wb@@VLTANO^CNFZHVJJDT@L?d@IRWXUDu@Qg@DYCOWMIKK[MYUUa@SOSEc@A_@CYBKJYZUXURG?YSEEWKY][QUc@Oa@OIIKQO[SQWUYWWO]Ia@Aa@@MCWISOWO[S]WMWCSOa@G[OUU[K[SSQSUUOUAWJ[F]?i@MKg@Bc@Aa@AEBe@?WBYLg@Ve@NYFUT]Pa@HUB]@a@Am@Ag@S_@QQYUS]EIQg@M]MWMWMOQQIKG?C?OES?OMDGFK?CMDIQEMMOKa@Q]G]MOM]@UCBC[Ea@LWLW@UDWH_@DYU[CWUKYKY@OBSFWBWDWESCWEU?QB[?OFQJSRQPQ@MBOCYB]EWISBWKEICKKC?NHKIL?GK?UCOBQ@QKIEOA??KFOEKAO?K?UKMYU_@K[OUQY?a@C[AI@WKGKOGIBYKOQOAWH?MKIW?UGGKDSJQDK@MBC@KHABK@KDQNMF?BQKL@GJWJWHOHYPMVO?CQMQWEAKGMO?HKKGCDMWFIGECKSEIQGKOUKIOMIOCQCKGQGISQA?QUM[EGOIOEOKIQOICGKEQIGGKIQOMMIQIK?MIGOKOKC?EKQO[WG@AEESBGOMMECCEAIIKGOCOBMFKAICOAI?B@EOBB???CGAGKEIOS@_@EGGQEEGUIIEGGEEUG?@DOKBICGHFIKEEEKNA@J?LOM@CAE??GOGGAKF@BD@?FD@L@@OBF@@?FHB@BB@D?D??DF?BCCVNFHB@DHHJJRDTDTHVFPFNDFFABJHCD?FBL?NCPBLHD?DDB?JJ?EFLFD??@?CDJBB@AB@DD?LPVNLLPLPLDDHHFJFNHXTPLLNF??JDNJPBHRFHB?LHPPN\\TPNTPT@@FPFTFHLLNNNPRVDN@FFDFJDBDND?HJDDFFBHLJLTBPH?JOKBSP?@?BDCEFFGBBGENCAK@BA?GOBBEDAEBDF@HDK?LSTY\\KXKTGVO@?HEVKRKRCVERQVVFJFJBHBHDRHJ?ZFTFLLTJV@^Dh@Nh@NXPd@RTT?N?LDFBB?JCB?JDDBDCLHN?H?B?D@TB\\Jd@F\\L^?\\@F?LDVEVUTWZK^A`@EZ@RDd@BTIREZI??RH\\HVDVPNJLPHZ?\\?VBPM^GXCZEZGVF`@ANBRJf@BPNZFTJFJEFJCPGDEHTCPB@?THTL@P?BIEB??I?HLCR`@RZT`@DJRb@LZNVT\\J\\F`@Fd@D\\?ZEh@GPEVS`@E\\Of@QVQ^A^GXAX?ZE`@LZR?NCRAVE`@DTLZN\\Tf@VXPZLZBXB^PXRN`@PVTf@LJL`@Bh@?`@Nd@Pb@RZb@^ZZZZR`@T\\TJVZRRTDLMRa@ZWTKX?L@^B`@DXL@?R^X\\^V\\N`@F\\BNHV?RVTIAQJEBm@?O@WC[AK?EAY?I@WE]H_@LSLSV]Jc@@[Kc@Cc@Ae@K]QU?a@XB@BNn@Fr@Jh@HVBHBBDYEKHAE@AB@CC???B?A?A?CB???C??BBDC@?@TBh@Bd@Nl@LPH\\Hf@@VHZLXJ^DJB\\EVGd@AP?`@F\\Ff@@PHn@Fb@F`@@B@t@N|@FZXXRj@L~@Tb@DE?ICEBACL?JIACN??FCHHJP@@Xf@R\\Zf@FPRl@Nf@Nf@Ph@Tf@V\\^XVCTA\\ET?`@@RVGj@VFNBDQNF@?R`@NHNFLAJ@LB@YJAPDN@HBT?NCRNb@Ad@Sh@Mb@Ib@Gd@Cf@Ed@OFMLKRILGHPNDLERUL]TQ@CT]?CGQDEVWFIHERMNSR?BHAb@@X@?@L@JN?HKLEJZFLHHHBHUDK@?DQFSFSDUJ?BNDFLX?JFPJEJUJWFWFIFKLIFTENG\\ENDPDIBELQJELOLMLE@?@JCPGFABITCL?T@HLSJEJETMHCLECNIPAPJCJKHALKNQDCAJEPFJFAL?FKFOCGG?')
	# r = polyline.decode('}skwEmp_qY??')
	# r = decodeLevel('G@@AAAB@A@AB@AA@BAC@AAABAAA?B@AABC@A@AAB@AA@BAAB@AA@CAAABAAB@A@BACDAA@BA@AB@AABA@A??@CAA@AB@A@AB@A@@AB?@A@A@AC@AAAB@A@ABA?GB@A@CA@?AB@AA@BAA@AB@A?A@ACD@A@A@A@B@A@@A@AB@A@@A@G@@A@A@AA@BA@AB@A@ABA@AG')
	# r = decodeLevel('G@A@?A@@?@?@A@?@@A?@??@@@@A@@B@@A@@A@@?@@@A@@A@?A@AB@@?@A@A@A?@A@A@@@AB@A@??@A?@@?@A?@@A@@A@@?@@A?B?@??@A@@@A@AB@A@C@A@A@AB@A@A@@AB@@?@A?@?@@@A@@@A@B?@@A?A@A@A@C@G@?@A@ABAA@AB@AAB??@A@C????@A????????@????????????????????@??@A@A@A???@@@A@@A@?AB@?AAAB@@A@?????@@@?A@A@A@B@@A@A@AABAA@BA@A@@BC@A@@A@BAA@AB?@@@A@ABA?C@AG?@A@?ABA@AB@@@AAB@?@@A@AB@A?@?AB@A@AC@A@A@AB@AA@A@B@A@A@B@A@@A@B?@AGA@AAB@@A@AB@A@A@B@A@C@A@A@AB@A@A@A@BAA@AB@AAAB@AC@A@ABA@A@AB@A@A@AB?@@A@@?@A@C@A@AB@?A@A@B@A@A@B@A@@A@A@B@@A@@A@@B@@AD@@A@@@@@@@A@@A@B?@@@A@@AAB@A@A@B@@A@A@A?@B@C@@A@@?A?@@A@?B@@@A@A@@@A@@A@@@B@@A@@A@A@B@A@@A@?AB@@A@A@@@C@@A@A@@A@B?@A@?A@?A@?@B@@@A@?@A?@????@A@B@@A@@A?@??C@@@@?A@@@@?@?@@G?@?@@A@??@?@?@?A??@@A@B@A@A@A@?B?@?@A@@?A?@?@A???@??@?B@A@A?@@AC@A@?@A@@A?B@A@A@?B@@A@A@AB@@?A?@?A@@AC?@?@???@?@@@@??@?@???@?@A@A@B?@@A@@A@A@@@AB@A@A@BAA@B@A@@?@A?@??A@?@?BC@A@A?@AB@A@A@AB@A@?A@A@B@A@A@B@A@AB@AD@A@@@A@?A@@?B@A???@@@@AB@A@AB@AA@CA@AABA@A@B@A@A@@A@B@A@AB@A@C@A@AB@A@BA@AB@A@C@G@@A@A@@B@A?@AB@A@A@AB@@A@A@B?@A@A@B@AA@BCA@A@G?AAB@A?@@A?@?????????????@AAB@AA@B@A@ACA@A@B@AA@BAAB@ACA?@?@@@?@??@A?ABA@ABAAABG@A@A@AAB@@@?A@@A@@@A@@B@@A@AAB@A@A@C@@@A@A@A?B?@?A@@A@@@A@?A@@@@A@@B@A?@A@A@@@A@@@@AB@@A@A@A@@?@A@A@?@@@A@A@@@@@A?@@@B@@?A@?@@@?A@CGG')
	encoded_levels = [
"G@@AAAB@A@AB@AA@BAC@AAABAAA?B@AABC@A@AAB@AA@BAAB@AA@CAAABAAB@A@BACDAA@BA@AB@AABA@A??@CAA@AB@A@AB@A@@AB?@A@A@AC@AAAB@A@ABA?GB@A@CA@?AB@AA@BAA@AB@A?A@ACD@A@A@A@B@A@@A@AB@A@@A@G@@A@A@AA@BA@AB@A@ABA@AG",
"GG",
"G??G",
"GG",
"GA@ABAAABAAABA@C@@A@ABAA@B@A@@A?B@A@A@C@A@@A@AB@AA@AB@A@?@A@B@A@C?@A@A@B@AAB@A@AAB@A@A@D@?A@A@B@A@AB@?@?G@?A??@@?A@@AB@@A@A@@A@@@@@BA@A@A@B@A@@A@@?@@?@?@B?C@A@?@@@@@A@B@@@@A@@A?@A?B?@@A@@A@@@A@@A@B@@@?A@@A@@?AA@B@@A@??@C?@??@A@@?A@@?@A@B@A?@?@@@A@@?A@@?@@@B???@?@A@@@?@A@@@@@@?A@AB@@@?A@@A@@A?@@@@@?@@A@@@?@@?B@@A@C@@@A??@@@@??@A?G?@?@A?@A?@@A?@?@@?@@?@?@?@?@@@B@A?@??A@@?A@B@@A@@@A@@B@@@@A@??@A??@??A??@@BC?@?@A?@?@@??A@@?@G?@@A@?@@A?@?@A?@??@?@A?@?@B?@??@?A?@@?A?@A?@B@@A@A@@A@B?@?@A?@A@?@AB?@@A?@???@?@??@@@@AC@@?A@@A@B@@@?@@@A@A@@AB@A@A@@A@B@A@@A?@?@?A?@?AC@@?A@@@@A@@A@B@@A@?@?A@?@?A@B@?@A@@A@@A@B?@@???@??@?@?@?@A?@?@A?@?@@?@@??@?@@??@@?@A@@@A@@@B@C?@?A@@A@@A@B@A@G@@A@G@A?@@@AB@@?A@@A@?@A@B@@A@@A@@AB@A@A@@A@C?@A@@A@@??@?B@@A@@?A@?@?B@A@@@A@@@@A?B@@@@A@?@A@@AC@@@@@A@?A@@?AB@@A@A@?@A@?@@@A@@??A?@B@??@A?@@?@A@?@A?@B@@?A@A@AC@A@A@@AB?@??@@@A?@A@@A?B?@@A@@A?@ABA@@A@@?A@@AC@DA@A@BA@A@AB@AAAB@AC@A@A@B@A@A?@@BA@A@B@A@A@?C@A@AAB@@A@A@AB@?A@A@BA@C?@A@A@B@A@AB?AAA?B@D@AAB@?G",
"GG",
"G?G",
"GG",
"G@@@AA@@B?@AA@B@A@AB@A@@A@A@A?@@?A?@C@A@A@B@A?@A@@AB@A@AB?@A@C@A@A@AB@AA@B?AABAACAAAB@A@AB@AA?AB@A@A@A@ACD@A@BAAA@B@AABAACAAABAAB@AAABAC?A@A@@@@@A@@?@A?@???@@?@?@@@?@@@@??@A@@@@AA@A@ABAAABA@A@B@AC@A@ABAA@B@A@AAB@AD?AAABAAABG@ABAACAA@AB@AA@ABA@A@??AB@A@@@@??GG"
	]

	r = []
	num = 0
	for lev in encoded_levels:
		item = decodeLevel(lev)
		num += len(item)
		r.append(item)
	print(r, num)

if __name__ == '__main__':
	main()
