
import argparse
import os
import re
import datetime
import shutil

YAMAP_GPX_DIRS = ['M:\\down\\yama', 'C:\\down\\yamap']	# YAMAPからdownloadしたGPXファイルが置かれているディレクトリ

def usage():
	print('''
parameter should be YYYYMMDD or YYYYMMDD-YYYYMMDD
''')

def main():
	MaxDays = 10

	parser = argparse.ArgumentParser(description="t2g3prep")
	parser.add_argument("date", nargs="?", help='YYYYMMDD or YYYYMMDD-YYYYMMDD')
	parser.add_argument("-m", '--max_days', help='maximum days')
	args = parser.parse_args()

	# 日付 指定が無ければ入力
	date = args.date
	if not date:
		print("Input date (YYYYMMDD)")
		now = datetime.datetime.now()
		date = "{:04}{:02}{:02}".format(now.year, now.month, now.day)
		print("Use " + date + " if only enter")
		r = input()
		if r != "":
			date = r

	print(date)	# 対象ディレクトリ

	if args.max_days:
		MaxDays = args.max_days

	m = re.match(r'^(\d\d\d\d)(\d\d)(\d\d)', date)
	if not m:
		usage()
		exit(1)

	start = datetime.datetime(year=int(m.group(1)), month=int(m.group(2)), day=int(m.group(3)))
	end = start
	m = re.match(r'.+-(\d\d\d\d)(\d\d)(\d\d)', date)
	if m:
		end = datetime.datetime(year=int(m.group(1)), month=int(m.group(2)), day=int(m.group(3)))

	# 複数のdirectory対応
	dirs = []
	dt = start
	while dt <= end:
		dirs.append(dt.strftime('%Y%m%d'))
		dt += datetime.timedelta(days=1)

	if len(dirs) > MaxDays:
		print("Too long days")
		exit(10)

	for dir in dirs:
		print(dir)

		if os.path.exists(dir):
			print("Warning: Directory {} already exists.".format(dir))
			continue

		os.mkdir(dir)

	# すでにbuild_xxxxが定義されているか、dirs内のそれぞれのdirectoryについて調べる
	dir_defined = {}
	for line in open('Makefile'):
		m = re.match(r'build_(\d+):', line)
		if m:
			# m.group(1)がdirs内のdirectoryと一致するか？
			dir = m.group(1)
			if dir in dirs:
				dir_defined[dir] = True
				print(f'Warning: build_{dir} is already defined. -> SKIPPED')
			else:
				dir_defined[dir] = False
		elif line.startswith('BUILDNEWS ='):
			# BUILDNEWSの中に、dirs内のdirectoryがあるか？
			for dir in dirs:
				if dir in line:
					dir_defined[dir] = True
					print(f'Warning: build_{dir} is already defined in BUILDNEWS. -> SKIPPED')

	modified = False	# TrueのときのみMakefileを変更保存
	with open('Makefile.new', 'w') as fw:
		b_mode = False
		b_mode_follow = False
		for line in open('Makefile'):
			if b_mode:
				if not b_mode_follow:
					dir = dirs[0]
					line, n = re.subn(r' build_(\d+)', ' build_'+dir, line)	# 指定のdirectoryに変更
					if n == 0:
						print("Failed to find target b")
						exit(1)
					b_mode_follow = True
					fw.write(line)
					if not is_defined(dir, dir_defined):
						fw.write('\nbuild_{}:\n'.format(dir))
						for dir in dirs:
							fw.write('	$(MAKE) -DDIR={} buildnew\n'.format(dir))
							modified = True
					continue
				# 複数日対応の場合
				if line != '\n':
					# 非改行
					continue
				# 改行のみ
				b_mode = False
				b_mode_follow = False
	
			if line.startswith('b:'):
				b_mode = True

			# BUILDNEWSへの追加
			if line.startswith('BUILDNEWS ='):
				line = line.rstrip('\n')
				for dir in dirs:
					if not is_defined(dir, dir_defined):
						line += ' build_' + dir
						modified = True
				line += '\n'

			fw.write(line)

	if modified:
		if os.path.exists('Makefile.old'):
			os.unlink('Makefile.old')
		os.rename('Makefile', 'Makefile.old')
		os.rename('Makefile.new', 'Makefile')
		print("Makefile is changed.")

	# GPXファイルのcopy
	for dir in dirs:
		if os.path.exists(dir):
			mm = re.match(r'(\d\d\d\d)(\d\d)(\d\d)', dir)
			if not mm:
				continue
			y = mm.group(1)
			m = mm.group(2)
			d = mm.group(3)
			yamap_gpx_prefix = f'yamap_{y}-{m}-{d}'
			for yamap_dir in YAMAP_GPX_DIRS:
				if not os.path.exists(yamap_dir):
					continue
				gpx_files = [f for f in os.listdir(yamap_dir) if f.startswith(yamap_gpx_prefix) and f.endswith('.gpx')]
				# gpx_file: 'yamap_YYYY-MM-DD_hh_mm.gpx
				# ex. yamap_2025-06-17_07_59.gpx
				if len(gpx_files) > 0:
					src = os.path.join(yamap_dir, gpx_files[0])
					dst = os.path.join(dir, f'{dir}.gpx')
					if not os.path.exists(dst):
						shutil.copy2(src, dst)
						print("Copy {} to {}".format(src, dst))
						break
					else:
						print("Warning: {} already exists.".format(dst))

	# 温度データファイルのrename
	# ex. 防水温湿度計 41_data.csv → temphumid.csv
	for dir in dirs:
		if os.path.exists(dir):
			filename = '防水温湿度計 41_data.csv'
			if os.path.exists(os.path.join(dir, filename)):
				new_filename = 'temphumid.csv'
				if os.path.exists(os.path.join(dir, new_filename)):
					print("Warning: {} already exists.".format(new_filename))
					continue
				# rename
				os.rename(os.path.join(dir, filename), os.path.join(dir, new_filename))
				print("Rename {} to {}".format(filename, new_filename))

def is_defined(dir, dir_defined):
	if dir in dir_defined:
		return dir_defined[dir]
	else:
		return False

main()
