# 指定したファイル名（の一部）のリストをYAMAPの編集画面の写真へ一括アップロード
# ファイル名リストはクリップボード経由

import argparse
from contextlib import nullcontext
import os
import time
import datetime
import glob
import re

import pyperclip
import pyautogui as gui
import msvcrt
from PIL import ImageGrab

# DefaultDirectory = r'T:\HPDrive\album'
DefaultDirectory = ''
MaxFileSize = 0xA00000
Verbose = False
HPMode = False	# HPだと特殊な動きをするため
ScrollForUpload = -400

def main():
	parser = argparse.ArgumentParser(description="yamup - upload to yamap")
	parser.add_argument("date", nargs="?", help='YYYYMMDD or YYYYMMDD-YYYYMMDD')
	parser.add_argument('-d', '--dir', help='directory for photo files')
	parser.add_argument('-dd', '--defaultdir', help='default directory for photo files')
	parser.add_argument('-c', '--clipboard', help='use clipboard for input file name list')
	parser.add_argument('-p', '--pause_each_photo', action='store_true', help='Upload with pause on each photo')
	parser.add_argument('-m', '--upload_msg', action='store_true', help='Upload messages too')
	parser.add_argument('-v', '--verbose', action='store_true', help='verbose for debug')
	parser.add_argument('-hp', '--hp', action='store_true', help='HP mode')
	parser.add_argument('-t', '--test', action='store_true', help='test for debug')
	args = parser.parse_args()

	files = []
	comments = {}

	if args.verbose:
		global Verbose
		Verbose = args.verbose

	if args.hp:
		global HPMode
		HPMode = True

	if args.clipboard:
		s = pyperclip.paste()
		files = s.split('\r\n')

	if args.defaultdir:
		global DefaultDirectory
		DefaultDirectory = args.defaultdir

	# 対象ディレクトリの設定
	dir = args.dir
	if dir:
		if not os.path.isabs(dir):
			dir = os.path.join(DefaultDirectory, dir)
	else:
		date = args.date
		if not date:
			now = datetime.datetime.now()
			date = "{:04}{:02}{:02}".format(now.year, now.month, now.day)
			print("Use " + date + " if only enter")
			r = input()
			if r != "":
				date = r
		dir = os.path.join(DefaultDirectory, date)

	if not args.clipboard:
		if args.upload_msg:
			file = os.path.join(dir, 'photomarkcomments.txt')
			if os.path.exists(file):
				files, comments = parsePhotomarkCommentFile(file)
				if Verbose:
					print(files)
					print(comments)
			else:
				print("No photomarkcomments.txt in "+dir)

		if not files:
			file = os.path.join(dir, 'photomarks.txt')
			if not os.path.exists(file):
				print("No photomakrs.txt in "+dir)
				return
			for line in open(file):
				line = line.rstrip('\n')
				files.append(line)

	global ScrollForUpload
	if args.upload_msg:
		ScrollForUpload = -200	# 画面の大きさに応じて微調整が必要

	print("Find files in "+dir)
	allfiles = glob.glob(dir + '\\*.jpg')

	# ファイル名とファイルサイズの検証
	ok = True
	upfiles = []
	for file in files:
		if file == "":
			continue
		found = None
		for afile in allfiles:
			_d, basename = os.path.split(afile)
			if file in basename:
				found = afile
				upfiles.append(found)
		if not found:
			print("{} : Not found!!".format(file))
			ok = False
		else:
			size = os.path.getsize(found)
			if size >= MaxFileSize:
				print("{} : {:,} bytes. size over!!".format(found, size))
				ok = False
			else:
				print("{} : OK".format(found))

	if not ok:
		return

	print("File check is ok. ({} files)".format(len(upfiles)))
	print("")

	print("Activate YAMAP page and show the edit page.")
	print("Move the mouse cursor to the YAMAP page.")
	print("If you are ready, press enter.")

	if args.test:
		pos = findUpIcon()
		print(pos)
		wait_upload(pos)
		exit(0)

	wait(2)

	while True:
		input()
		pos = findUpIcon()
		if pos:
			if Verbose:
				print("pos=",pos)
			break
		print("Not found upload icon.")
		print("Scroll the page to show the upload icon.")

	upload_icon_pos_origin = findUpIcon()

	for i, file in enumerate(upfiles):
		print("uploading... {} {}/{}".format(file, i+1, len(upfiles)))

		# upiconが出るまでscroll
		if Verbose:
			print("Scrolling")
		upload_icon_pos = findUpIconScroll()
		if not upload_icon_pos:
			exit(1)
		print(f"click upload icon: {upload_icon_pos[0]}, {upload_icon_pos[1]}") if Verbose else None
		gui.click(upload_icon_pos[0], upload_icon_pos[1])

		wait(1.5)

		# ファイル名の入力
		if Verbose:
			print("Entering filename")
		fullpath = os.path.abspath(file)
		gui.typewrite(fullpath)
		wait(1.5)
		# Alt+O, Enterの片方だけだと失敗するときがあるため
		gui.hotkey('alt', 'o')
		if not HPMode:
			gui.press('enter')

		if args.upload_msg:
			wait_upload(upload_icon_pos)
			_d, filename = os.path.split(file)
			basename, ext = os.path.splitext(filename)
			if basename in comments and comments[basename] != '':
				column = i % 3
				print(f"Enter to edit comment column={column}") if Verbose else None
				while True:
					upload_icon_pos = findUpIconScroll()
					if not upload_icon_pos:
						print("No upload icon")
						return
					scroll_to_origin(upload_icon_pos, upload_icon_pos_origin)
					pos = click_comment(column, upload_icon_pos)
					wait(0.3)
					if not pos:
						print("retry to locate the upload icon")
						continue
					comment = comments[basename]
					print("comment=", comment)
					# gui.typewrite(comment)
					pyperclip.copy(comment)
					gui.hotkey('ctrl', 'v')
					wait(0.3)
					break
			upload_icon_pos = findUpIconScroll()
			if not upload_icon_pos:
				return
			scroll_to_origin(upload_icon_pos, upload_icon_pos_origin)

		if args.pause_each_photo:
			input("Enter if uploaded.")
		else:
			if not args.upload_msg:
				# ７秒wait
				timeout = 7
				print("If uploaded, then press any key or wait {} seconds".format(timeout))
				for i in range(timeout):
					print("{} ".format(timeout-i), end='\r')
					wait(1)
					if msvcrt.kbhit():
						k = msvcrt.getch()
						break

	print("Uploaded {} files".format(len(upfiles)))

def scroll_to_origin(icon_pos, icon_pos_origin):
	diff = int(icon_pos[1] - icon_pos_origin[1])	# 初期位置とのずれ
	if diff > 10:
		print("Scroll: ", diff)
		gui.scroll(-diff)	# 画像追加によるuploading red barが消える場合
		wait(0.3)

class MultiIcon:
	def __init__(self, iconlist):
		self.iconList = iconlist
		self.iconPaths = []
		self.lastIcon = None
		for file in iconlist:
			path = get_file_path(file)
			self.iconPaths.append(path)

	def locate(self, region=None):
		if not self.lastIcon:
			for path in self.iconPaths:
				pos = locateOnScreen(path, region=region)
				if pos:
					print("Found with", path, pos)
					self.lastIcon = path
					return pos
			return None
		else:
			return locateOnScreen(self.lastIcon, region=region)

UpIcons = None
UploadingIcons = None
PhotoIcons = None

def findUpIconScroll():
	pos = None
	i = 0
	scrolled = False
	while True:
		pos = findUpIcon()
		if pos:
			if scrolled:
				# ギリギリだとmiss hitする可能性がありそうなのでもうちょっとscrollさせる
				if Verbose:
					print("Scroll small up")
				gui.scroll(-100)
				wait(0.5)
				pos = UpIcons.locate()
				if not pos:
					print("Error: Why scroll out??")
					return None
			break

		if Verbose:
			print("Scroll")
		gui.scroll(ScrollForUpload)
		i += 1
		if i > 20:
			print("Error: scroll timeout!!")
			return None
		scrolled = True
		wait(0.5)

	return pos

def findUpIcon():
	global UpIcons
	if not UpIcons:
		upicons = ['yamup-up1.png', 'yamup-up2.png', 'yamup-up4.png']
		UpIcons = MultiIcon(upicons)

	return UpIcons.locate()

def wait_upload(upload_icon_pos, wait_uploading=True):
	global UploadingIcons
	if not UploadingIcons:
		redbaricons = ['yamup-redbar1.png', 'yamup-redbar2.png', 'yamup-redbar3.png']
		UploadingIcons = MultiIcon(redbaricons)
	
	region = None
	if upload_icon_pos:
		# <-- 400 --- upload icon --- 
		# ↓100
		# redbar ----------------------------
		pos = upload_icon_pos
		if HPMode:
			region = (pos[0]-600, pos[3]+100, pos[0]-400+200, pos[3]+100+300)
		else:
			region = (pos[0]-400, pos[3]+100, pos[0]-400+200, pos[3]+100+200)
		# print("region=", region)

	uploading = False
	if not wait_uploading:
		uploading = True

	while True:
		pos = UploadingIcons.locate(region=region)
		if not pos:
			if uploading:
				print("Uploaded.")
				return
			continue
		uploading = True
		print(f"Uploading... {pos=}")
		# gui.moveTo(pos[0], pos[1])
		wait(0.5)

def click_comment(column, upload_icon_pos):
	pos = find_photoicon(column, upload_icon_pos)
	if not pos:
		return None
	print("click: ", pos[0], pos[1]-30)
	gui.click(pos[0], pos[1]-30)
	return pos

def find_photoicon(column, upload_icon_pos):
	global PhotoIcons
	if not PhotoIcons:
		photoicons = ['yamup-photoicon1.png', 'yamup-photoicon2.png', 'yamup-photoicon3.png', 'yamup-photoicon4.png']
		PhotoIcons = MultiIcon(photoicons)

	region = None
	if upload_icon_pos:
		# ---------- photos -----------------
		# ↑200 or over
		# <-- 400 --- upload icon --- 400 -->
		pos = upload_icon_pos
		pos = (pos[0]-400, pos[1]-200, pos[2]+400, pos[1])
		# columnに応じて検索範囲を絞る
		if HPMode:
			region = (pos[0] + column*300, pos[1], pos[0] + column*350+400, pos[3])
		else:
			region = (pos[0] + column*300, pos[1], pos[0] + column*300+400, pos[3])

	while True:
		if region:
			print(f"region={region} column={column}")
		pos = PhotoIcons.locate(region)
		print("photoicon: pos=", pos)
		if pos or not region:
			return pos

		# region指定の場合は領域を少しずつ上に広げる
		region = (region[0], region[1]-200, region[2], region[3])
		

def get_file_path(filename):
	d, base = os.path.split(__file__)
	path = d + '/' + filename
	if not os.path.exists(path):
		print("ERROR: Not found up icon file. {}".format(path))
		exit(1)
	return path

def parsePhotomarkCommentFile(file):
	files = []
	comments = {}
	name = ''
	comment = ''
	for line in open(file):
		r = re.match(r'(\w+_\d+)', line)
		if r:
			# photo name
			name = r.group(1)
			files.append(name)
			comments[name] = ''
		else:
			line = line.rstrip('\n')
			if line == '':	# 空行
				if name:
					comments[name] = comment
					comment = ''
			else:
				if comment != '':
					comment += '\n'
				comment += line

	return files, comments

def wait(delay=0.5):
	time.sleep(delay)

def locateOnScreen(filename, region=None, grayscale=False, tmpfile=None):
	if not tmpfile:
		tmpfile = '/temp/cap_yamup.png'
#	win32cap.screenshot(tmpfile, region=region)	# Python 3.10ではwin32capがない？？
	ImageGrab.grab(bbox=region).save(tmpfile)
	pos = locateOnImage(filename, tmpfile, grayscale)
	if pos:
		if region:
			return (pos[0]+region[0], pos[1]+region[1], pos[2]+region[2], pos[3]+region[3])
	return pos

# locateOnScreen()はmulti monitor非対応なので車輪の再発明
def locateOnImage(objfile, targetfile, grayscale=False):
	import cv2
	import numpy as np
	img = cv2.imread(targetfile)
	if grayscale:
		img_gray = cv2.cvtColor(img, cv2.COLOR_BGR2GRAY)

	template = cv2.imread(objfile)
	if grayscale:
		template_gray = cv2.cvtColor(template, cv2.COLOR_BGR2GRAY)

	# 処理対象画像に対して、テンプレート画像との類似度を算出する
	if grayscale:
		res = cv2.matchTemplate(img_gray, template_gray, cv2.TM_CCOEFF_NORMED)
	else:
		res = cv2.matchTemplate(img, template, cv2.TM_CCOEFF_NORMED)

	# 類似度の高い部分を検出する
	threshold = 0.8
	loc = np.where(res >= threshold)
	if len(loc[0]):
		if True:
			# HPとPC7ではxとyが逆になっている？？
			y = loc[0][0]
			x = loc[1][0]
		else:
			x = loc[0][0]
			y = loc[1][0]
		if Verbose:
			print(len(loc), x, y)
		h, w, d = template.shape
		return (x, y, x+w, y+h)
	return None

main()
