#
# Sakura server上にあるphoto image fileをdownloadする
#
#

import datetime
import re
import os
import requests
import exifread
import argparse

def main():
	parser = argparse.ArgumentParser(description="getalbum")
	parser.add_argument("date", nargs="?", help='YYYYMMDD or YYYYMMDD-YYYYMMDD')
	parser.add_argument("-s", "--skip", action="store_true", help="skip existing files")
	parser.add_argument("-r", "--reverse", action="store_true", help="Download in reverse order")
	parser.add_argument("-n", "--max_num_files", type=int, help="maximum number of files to download", default=-1)
	args = parser.parse_args()

	date = args.date
	if not date:
		print("Input date (YYYYMMDD)")
		now = datetime.datetime.now()
		date = "{:04}{:02}{:02}".format(now.year, now.month, now.day)
		print("Use " + date + " if only enter")
		r = input()
		if r != "":
			date = r

	print(date)

	m = re.match(r'^(\d\d\d\d)(\d\d)(\d\d)', date)
	if not m:
		usage()
		exit(1)

	start = datetime.datetime(year=int(m.group(1)), month=int(m.group(2)), day=int(m.group(3)))
	end = start
	# m = re.match(r'.+-(\d\d\d\d)(\d\d)(\d\d)', date)
	# if m:
	# 	end = datetime.datetime(year=int(m.group(1)), month=int(m.group(2)), day=int(m.group(3)))

	dirs = []
	dt = start
	while dt <= end:
		dirs.append(dt.strftime('%Y%m%d'))
		dt += datetime.timedelta(days=1)

	for dir in dirs:
		print(dir)
		if not os.path.exists(dir):
			os.makedirs(dir)

		url = 'https://pdic.sakura.ne.jp/cgi-bin/upload/upload.cgi?action=list&album=1&date='+dir
		urls, files = list_files(url)
		if files:
			urlbase = 'https://pdic.sakura.ne.jp'
			total = len(files)
			if args.max_num_files > 0 and total > args.max_num_files:
				total = args.max_num_files
			files.sort()
			if args.reverse:
				files = files[::-1]
			for i, file in enumerate(files):
				if i >= total:
					break
				filename = os.path.join(dir, file)
				if args.skip and os.path.exists(filename):
					print("Skip: "+filename)
					continue
				print(f'Downloading {i+1}/{total}: {filename}')
				download_file(urlbase + urls[file], filename)

		# if os.path.exists(dir):
		# 	print("Warning: Directory {} already exists.".format(dir))
		# 	continue

		# os.mkdir(dir)

def list_files(url):
	print("url=",url)
	ret = url.split('?')
	data = {}
	if len(ret) > 1:
		data = parse_params(ret[1])
	res = requests.post(url, data=data)
	if res.status_code != 200:
		print("error: "+url)
		return None, None
	# print(f'list: {res.text}')
	urls = {}
	files = []
	lines = res.text.splitlines()
	for line in lines:
		m = re.search(r'^<a href="(.+)">(.+)</a>', line)
		if m:
			# print(m.group(1))
			url = m.group(1)
			file = m.group(2)
			base, ext = os.path.splitext(file)
			if ext.lower() == '.jpg' or ext.lower() == '.txt' or ext.lower() == '.gpx' or ext.lower() == '.kml' or ext.lower() == '.csv':		# download対象のファイル
				urls[file] = url
				files.append(file)
	print(files)
	return urls, files

def download_file(url, filename):
	data = requests.get(url).content
	with open(filename, 'wb') as f:
		f.write(data)
	if filename.lower().endswith('.jpg'):
		timestamp = get_exif_timestamp(filename)
		if timestamp:
			set_timestamp(filename, timestamp)
	# print("Downloaded: "+filename)

def parse_params(query):
	data = {}
	params = query.split('&')
	for param in params:
		pair = param.split('=')
		key = pair[0]
		if len(pair) > 1:
			data[key] = pair[1]
		else:
			data[key] = ''
	return data

def get_exif_timestamp(jpeg_file):
	with open(jpeg_file, 'rb') as f:
		tags = exifread.process_file(f)
		if 'EXIF DateTimeOriginal' in tags:
			return tags['EXIF DateTimeOriginal'].values
		elif 'Image DateTime' in tags:
			return tags['Image DateTime'].values
		else:
			return None
		
def set_timestamp(jpeg_file, timestamp):
	# timestampをYYYY:MM:DD HH:MM:SS形式に変換
	timestamp = timestamp.replace(' ', ':')
	timestamp = timestamp.replace('-', ':')
	
	# timestampをdatetime型に変換
	timestamp = datetime.datetime.strptime(timestamp, "%Y:%m:%d:%H:%M:%S")
	
	# timestampをファイルのtimestampに設定
	os.utime(jpeg_file, (timestamp.timestamp(), timestamp.timestamp()))


def usage():
	print("")

main()
